import os

# deployment types available right now
from ticket_parser2.ticket_parser2_py import BlackboxClientId
from tvm2.protocol import BLACKBOX_MAP

QLOUD = 'qloud'
DEVELOPMENT = 'development'
NIRVANA = 'nirvana'

# determine where we are
if 'QLOUD_ENVIRONMENT' in os.environ:
    DEPLOYMENT_TYPE = QLOUD
    QLOUD_ENVIRONMENT = os.environ['QLOUD_ENVIRONMENT']
else:
    DEPLOYMENT_TYPE = DEVELOPMENT
    QLOUD_ENVIRONMENT = None

if 'NIRVANA' in os.environ:
    DEPLOYMENT_TYPE = NIRVANA

from pymongo.read_preferences import ReadPreference, Secondary

import logging
import yenv
import pytz

os.environ.setdefault("REQUESTS_CA_BUNDLE", '/etc/ssl/certs/ca-certificates.crt')

BASE_DIR = os.path.dirname(os.path.abspath(__file__))
STATIC_DIR = os.path.join(BASE_DIR, 'jafar/static')

# SECURITY WARNING: keep the secret key used in production secret!
os.environ.setdefault('SECRET_KEY', '_________')
SECRET_KEY = os.environ.get('SECRET_KEY')

DEBUG = yenv.choose_key_by_type({
    'development': True,
    'production': False,
}, fallback=True)

PROPAGATE_EXCEPTIONS = True  # handle exceptions by root logger

# Main I/O

# Read only mode: jafar doesn't write anythong to mongo.
READ_ONLY = False

mongodb_uri_template = 'mongodb://{user}:{password}@{hosts}/{database}?maxPoolSize=4&connect=false'

MONGO_MDB_HOSTS = yenv.choose_key_by_type({
    'production': [
        'vla-8h8moxom5mcqvhxq.db.yandex.net:27018',
        'sas-350atkewrtqdqznw.db.yandex.net:27018',
        'iva-g4rokm43mg6rjizc.db.yandex.net:27018',
    ],
    'testing': [
        'sas-m6ed1vwh01lxm0td.db.yandex.net:27018',
        'vla-5xqt1wmnesk3vqto.db.yandex.net:27018',
    ],
}, fallback=True)

MONGO_REPLICA_HOSTS = [
    'vla-8h8moxom5mcqvhxq.db.yandex.net:27018',
    'sas-350atkewrtqdqznw.db.yandex.net:27018',
    'iva-g4rokm43mg6rjizc.db.yandex.net:27018',
]

MONGO_USER = os.environ.get('MONGO_USER')
MONGO_PASSWORD = os.environ.get('MONGO_PASSWORD')
MONGO_REPLICA_PASSWORD = os.environ.get('MONGO_REPLICA_PASSWORD')

# jafar db (specific to this project)
MONGO_DBNAME = 'jafar_db'
MONGO_URI = mongodb_uri_template.format(
    user=MONGO_USER,
    password=MONGO_PASSWORD,
    hosts=','.join(MONGO_MDB_HOSTS),
    database=MONGO_DBNAME,
)
MONGO_READ_PREFERENCE = ReadPreference.PRIMARY_PREFERRED

# advisor db (general app recommender database with user and app info)
ADVISOR_MONGO_DBNAME = 'advisor_db'
ADVISOR_MONGO_URI = mongodb_uri_template.format(
    user=MONGO_USER,
    password=MONGO_PASSWORD,
    hosts=','.join(MONGO_MDB_HOSTS),
    database=ADVISOR_MONGO_DBNAME,
)

ADVISOR_MONGO_READ_PREFERENCE = ReadPreference.NEAREST

# advisor's replica
ADVISOR_MONGO_REPLICA_DBNAME = 'advisor_db'
ADVISOR_MONGO_REPLICA_URI = mongodb_uri_template.format(
    user=MONGO_USER,
    password=MONGO_REPLICA_PASSWORD,
    hosts=','.join(MONGO_REPLICA_HOSTS),
    database=ADVISOR_MONGO_REPLICA_DBNAME,
)
ADVISOR_MONGO_REPLICA_READ_PREFERENCE = Secondary(tag_sets=[{"use": "replica"}])

# Jafar replica
JAFAR_MONGO_REPLICA_URI = mongodb_uri_template.format(
    user=MONGO_USER,
    password=MONGO_REPLICA_PASSWORD,
    hosts=','.join(MONGO_REPLICA_HOSTS),
    database=MONGO_DBNAME,
)
JAFAR_MONGO_REPLICA_READ_PREFERENCE = Secondary(tag_sets=[{"use": "replica"}])

MONGODB_SETTINGS = [  # for mongoengine
    {
        'db': MONGO_DBNAME,
        'host': MONGO_URI,
        'read_preference': MONGO_READ_PREFERENCE
    },
    {
        'alias': 'advisor',
        'db': ADVISOR_MONGO_DBNAME,
        'host': ADVISOR_MONGO_URI,
        'read_preference': ADVISOR_MONGO_READ_PREFERENCE
    },
    {
        'alias': 'advisor_primary',
        'db': ADVISOR_MONGO_DBNAME,
        'host': ADVISOR_MONGO_URI,
        'read_preference': ReadPreference.PRIMARY
    },
]

MONGODB_REPLICA_SETTINGS = [  # for mongoengine when reading replicas
    {
        'db': MONGO_DBNAME,
        'host': JAFAR_MONGO_REPLICA_URI,
        'read_preference': JAFAR_MONGO_REPLICA_READ_PREFERENCE
    },
    {
        'alias': 'advisor',
        'db': ADVISOR_MONGO_REPLICA_DBNAME,
        'host': ADVISOR_MONGO_REPLICA_URI,
        'read_preference': ADVISOR_MONGO_REPLICA_READ_PREFERENCE
    }
]

# Stats clickhouse
CLICKHOUSE_CLUSTER = yenv.choose_key_by_type({
    'testing': 'advisor_clickhouse_test',
    'production': 'mdbuj5chhcg3tkkt8rh3'
}, fallback=True)
CLICKHOUSE_PORT = 19000
CLICKHOUSE_DATABASE = 'statsdb'
CLICKHOUSE_USER = os.environ.get('CLICKHOUSE_USER')
CLICKHOUSE_PASSWORD = os.environ.get('CLICKHOUSE_PASSWORD')
CLICKHOUSE_COMPRESSION = 'lz4'
CLICKHOUSE_CONNECTION_POOL = os.environ.get('CLICKHOUSE_CONNECTION_POOL', 100)
CLICKHOUSE_SETTINGS = {'max_threads': 1}
CLICKHOUSE_TABLE_PHONES = 'phone_collection'
CLICKHOUSE_USAGE_LOGS_TABLE = 'usage_logs'
CLICKHOUSE_USAGE_HOURLY_TABLE = yenv.choose_key_by_type({  # TODO: Get rid of this when tables have same names!
    'production': 'usage_hourly_new',
    'testing': 'usage_hourly',
}, fallback=True)
CLICKHOUSE_USAGE_WEEKLY_TABLE = yenv.choose_key_by_type({
    'production': 'usage_weekly_new',
    'testing': 'usage_weekly',
}, fallback=True)
CLICKHOUSE_USAGE_COUNTERS_TABLE = 'usage_counters'

# localization db (translations and experiment configs)
LOCALIZATION_MONGO_DBNAME = 'localizations_db'
LOCALIZATION_MONGO_URI = mongodb_uri_template.format(
    user=MONGO_USER,
    password=MONGO_PASSWORD,
    hosts=','.join(MONGO_MDB_HOSTS),
    database=LOCALIZATION_MONGO_DBNAME,
)

CACHE_TYPE = 'simple'

SHARED_CACHE_REDIS_HOST = 'localhost'
SHARED_CACHE_REDIS_PORT = '6301'
SHARED_CACHE_REDIS_PASSWORD = os.environ.get('SHARED_REDIS_AUTH', '')
SHARED_CACHE_REDIS_DB = yenv.choose_key_by_type({'testing': 1, 'stress': 2, 'production': 3}, fallback=True)

# Recommender I/O

DATASET_PATH = yenv.choose_key_by_type({
    'development': os.path.join(os.path.expanduser('~'), '.jafar'),
    'production': '/var/lib/jafar/'
}, fallback=True) if DEPLOYMENT_TYPE != NIRVANA else os.environ['TMPDIR']

DEFAULT_DATASETS = [
    'advisor_mongo',
]

LOGGING = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'default': {
            'format': '[%(asctime)s] [%(process)d] [%(levelname)s] [%(name)s] %(message)s',
            'datefmt': '%Y-%m-%d %H:%M:%S',
            'class': 'logging.Formatter',
        },
        'stream': {
            'format': '[%(asctime)s] [%(levelname)s] %(message)s',
            'datefmt': '%H:%M:%S',
            'class': 'logging.Formatter',
        },
        'raw': {
            'format': '%(message)s',
            'datefmt': '',
        },
        'json': {
            '()': 'ylog.format.QloudJsonFormatter',
        },
    },
    'handlers': {
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'default',
        },
        'stream': {
            'class': 'logging.StreamHandler',
            'formatter': 'stream',
        },
        'qloud': {
            'class': 'logging.StreamHandler',
            'formatter': 'json',
        },
        'sentry': {
            'level': 'ERROR',
            'class': 'raven.handlers.logging.SentryHandler',
        }
    },
    'loggers': {
        '': {
            'level': yenv.choose_key_by_type({
                'development': logging.DEBUG,
                'testing': logging.DEBUG,
                'production': logging.INFO
            }, fallback=True),
            'handlers': yenv.choose_key_by_type({
                'development': ['console'],
                'stress': ['qloud'],
                'testing': ['qloud', 'sentry'] if DEPLOYMENT_TYPE == QLOUD else ['stream'],
                'production': ['qloud', 'sentry']
            }, fallback=True) if DEPLOYMENT_TYPE != NIRVANA else ['console', 'sentry'],
            'propagate': False
        },
        'stream': {
            'level': logging.INFO,
            'handlers': ['stream'],
            'propagate': True,
        },
        'yt.packages.requests': {
            'level': 'WARN'
        },
        'requests': {
            'level': 'WARN'
        },
        'nile': {
            'level': 'INFO'
        },
    }
}

# Other recommender settings

COUNTRIES = ('RU', 'US')

DEFAULT_COUNTRY = 'US'

# Use russian recommender model for our neighbors
RECOMMENDER_COUNTRY_MAP = {
    'BY': 'RU',
    'KZ': 'RU',
    'UA': 'RU',
    'TJ': 'RU',
    'AZ': 'RU',
    'UZ': 'RU',
}

TOP_N_COUNT = 200
MAX_UPDATED_AGE = 30 * 6

PRESERVE_CONTEXT_ON_EXCEPTION = False  # for flask-testing

QLOUD_OAUTH_TOKEN = os.environ.get('QLOUD_OAUTH_TOKEN')
YT_TOKEN = os.environ.get('YT_TOKEN')
YQL_TOKEN = os.environ.get('YQL_TOKEN')
YT_PROXY = 'hahn.yt.yandex.net'
YT_RESERVE_PROXY = 'arnold.yt.yandex.net'
YT_POOL = 'advisor'
YT_TABLET_CELL_BUNDLE = 'advisor'

YT_METRIKA_PATH_1_DAY = '//logs/yandex-phone-metrika-mobile-log/1d'
YT_METRIKA_PATH_30_MIN = '//logs/yandex-phone-metrika-mobile-log/30min'
YT_MOBMETRIKA_PATH_1_DAY = '//logs/yandex-phone-metrika-mobile-log/1d'
YT_LAUNCHER_API_KEY = '37460'
YT_CONFIG = {
    'proxy': {'url': YT_PROXY},
    'token': YT_TOKEN
}
YT_PATH_USERS_FULL = '//home/advisor/users_full'
YT_PATH_PHONES = '//home/advisor/phone/collection'
YT_PATH_ITEMS = '//home/extdata/mobile/google_play/latest/{region}'
YT_PATH_PARSED_ITEMS = '//home/advisor/apps/parsed'
YT_PATH_ITEMS_DYNAMIC = '//home/advisor/apps/prepared_dynamic'
YT_PATH_DYNAMIC_LINK = '//home/advisor/apps/dynamic'
YT_PATH_APP_ICONS = '//home/advisor/apps/icons'
YT_PATH_APP_URLS = '//home/advisor/package_names'
YT_PATH_JAFAR = '//home/advisor/jafar/'
YT_PATH_POSTBACKS_STATS = '//home/advisor/money-reports/adnetwork-postbacks'
YT_PATH_PROMOEVENTS = '//home/logfeller/logs/advisor-promoevents-log/1d'
YT_PATH_ADNETWORK_STATS = '//home/advisor/adnetwork_stats'
YT_PATH_PROMO_LOG = '//home/advisor/promo-log/1d'
YT_PATH_TMP = '//home/advisor/tmp'
YT_VANGA_PATH = '//home/advisor/vanga/datasets'
YT_PHONE_ACTIVATED = '//home/advisor/reports/phone/dau_activated'
YT_PATH_SENDER_LIST = '//home/advisor/sender-list'

# PROPER settings
YT_PROPER_PATH = '//home/advisor/proper/metrics'
PROPER_USERS = 50000
PROPER_DAYS_INTERVAL = 7
PROPER_TOP = 25
PROPER_MIN_LAUNCHES = 25

YT_TOP_TABLE_TEMPLATE = '//home/extdata/aurora/extdata/mobile_stores/google_play/top_{country_code}'

APP_INFO_COLLECTION_INTERVAL = 7
APP_INFO_PARSED_TTL_DAYS = 30
APP_INFO_DYNAMIC_TTL_DAYS = 7
# bandits settings
APP_STATS_DAYS_INTERVAL = 14

# Google play regions corresponds to "gl" parameter in the web version of Google Play.
# You can find list of supported regions here:
# https://a.yandex-team.ru/arc/trunk/arcadia/zootopia/hub/mobile_stores/mining/google_play_apps.py#L537-547

GOOGLE_PLAY_REGIONS_MAP = {
    'RU': 'ru',
    'US': 'us',
    'TR': 'tr',
    'BY': 'by',
    'UA': 'ua',
    'KZ': 'kz',
    'BR': 'br',
    'MX': 'mx',
    'DE': 'de',
}

# We drop out apps where detected description language is not expected.
# As example, apps with chinese description in russian GPlay.
# Here we describe which languages are suitable for each country.

EXPECTED_LANGUAGES_MAP = {
    'RU': ['ru', 'en'],
    'US': ['en'],
}

NILE_WHEELS_DIR = os.path.join(BASE_DIR, 'jafar_yt', 'nile_wheels')

USAGE_STATS_SAVE_HORIZON = 14  # in days

# trending
TRENDING_INTERVALS_IN_DAYS = (14, 28)
TRENDING_COUNT_STATS = ('installs', 'launches')
TRENDING_MONGO_COLLECTION = 'trending_by_{}'
TRENDING_MONGO_FIELD = '{}_days_trend'

# smart_launch
VANGA_USERS = 30000
VANGA_IMPLICIT_NEGATIVES = 4
VANGA_DAYS_INTERVAL = 21
VANGA_DAYS_TEST = 7
VANGA_SEED_DAYS = 5
VANGA_TRAIN_TEST_RATIO = 0.9
VANGA_NEW_USER_DURATION = 7  # Length of the period within which we should count
# user as new and return default usage stats
VANGA_MIN_USERS_FOR_APP = 10  # Only apps with more than this amount of users appear in default counters
VANGA_GENERAL_DAYS = 14
VANGA_GENERAL_DYNAMIC_PATH = '//home/advisor/vanga/general_dynamic_2'
VANGA_GENERAL_STATIC_DIR = '//home/advisor/vanga/general_stats'
VANGA_GENERAL_TTL = 14
VANGA_USE_CLASSNAMES = True

# from https://bb.yandex-team.ru/projects/MOBILE/repos/monorepo/browse/launcher/android/launcher-app/src/com/yandex/
# launcher/statistics/ItemAnalyticsInfo.java?at=launcher%2Fandroid%2Flauncher-app%2Fnext-release#27
VANGA_BLACKLIST_PLACES = {
    'shtorka': ['recently', 'search'],
    'all_apps': ['color', 'category', 'header', 'list'],
    'homescreens': ['screen', 'dock', 'simple_folder', 'full_folder'],
    'widgets_list': [],
    'alice': []
}

ARRANGER_ALS_ITERATIONS = 40
ARRANGER_ALS_EMBEDDING_SIZE = 150
ARRANGER_NN_WEIGHTS_STORAGE_PATH = 'arranger_nn_weights.bin'
ARRANGER_DAYS = 14
ARRANGER_TOP_CLASS_DYNAMIC = {
    'RU': '//home/advisor/proper/top_class_name_ru',
    'US': '//home/advisor/proper/top_class_name_us',
}
ARRANGER_TOP_CLASS_STATIC = '//home/advisor/proper/top_class_name_static'
ARRANGER_TOP_CLASS_STATIC_TTL = 14

MAAS_URL = 'http://saas-searchproxy.yandex.net:17000/'

if DEPLOYMENT_TYPE == NIRVANA:
    SENTRY_URL_TEST = 'https://5a3a36a49af2404e81e36ca93d2f3638' \
                      '@sentry-test-proxy.t.yandex.net/512?verify_ssl=0&timeout=10'
    SENTRY_URL_PROD = 'https://01fd8d1de3d74339b7734afc093a19fd' \
                      '@sentry.iddqd.yandex.net/846?verify_ssl=0&timeout=10'
elif DEPLOYMENT_TYPE == QLOUD:
    SENTRY_URL_TEST = 'https://77dc2804323540bda0e24eee148d076b' \
                      '@sentry-test-proxy.t.yandex.net/507?verify_ssl=0'
    SENTRY_URL_PROD = 'https://148b2db57bc342219ded286ae00cc620' \
                      '@sentry.iddqd.yandex.net/844?verify_ssl=0'
else:
    SENTRY_URL_TEST = ''
    SENTRY_URL_PROD = ''

os.environ.setdefault('SENTRY_DSN', yenv.choose_key_by_type({
    'testing': SENTRY_URL_TEST,
    'production': SENTRY_URL_PROD,
    'stress': '',
    'development': '',
}, fallback=True))
SENTRY_DSN = os.getenv('SENTRY_DSN')

ROBOT_LAUNCHER_YT_LOGIN = 'robot-launcher-yt'

FALLBACK_CONFIG_FILENAME = '/etc/jafar-fallback/jafar_fallback.json'

if DEPLOYMENT_TYPE in (DEVELOPMENT, NIRVANA):
    MEMMAP_STORAGE_DIR = os.path.join(DATASET_PATH, 'memmap_storage')
else:
    # for qloud installations, ephemeral disk is used which provides some
    # persistency and allows the storage to survive re-deployment
    MEMMAP_STORAGE_DIR = '/ephemeral/memmap_storage'

# lazy mode opens a memmaped file form disk upon request, then closes it
# afterwards. this shortens application starting time but results in slower requests
MEMMAP_STORAGE_LAZY_MODE = 'lazy'
# eager mode opens all memmaped files and keeps them in memory. this may require tweaking
# ulimit -s, some pre-loading time and also causes problems with flask's runserver,
# therefore not recommended in development
MEMMAP_STORAGE_EAGER_MODE = 'eager'
# set default mode
MEMMAP_STORAGE_MODE = MEMMAP_STORAGE_LAZY_MODE

# oauth config
OAUTH_YANDEX = {
    'authorize_url': 'https://oauth.yandex-team.ru/authorize',
    'access_token_url': 'https://oauth.yandex-team.ru/token',
    'consumer_key': os.environ.get('OAUTH_KEY'),
    'consumer_secret': os.environ.get('OAUTH_SECRET'),
    'base_url': None
}

TANKER_OAUTH_TOKEN = os.environ.get('TANKER_OAUTH_TOKEN')

GEO_SUGGEST_URL = 'https://suggest-maps.yandex.ru/suggest-geo'

HTTP_GEOBASE_URL = yenv.choose_key_by_type({
    'production': 'http://geobase.qloud.yandex.ru',
    'stress': 'http://geobase-load.qloud.yandex.ru',
    'testing': 'http://geobase-test.qloud.yandex.ru',
}, fallback=True)

SNAPSHOT_VERSION = 3
SNAPSHOT_LIMIT = 20
SNAPSHOT_FOLDER = os.path.join(
    DATASET_PATH, 'snapshots'
)

SANDBOX_OAUTH_TOKEN = os.environ.get('SANDBOX_OAUTH_TOKEN')

ADMIN_LOGO = yenv.choose_key_by_type({
    'production': 'Jafar Prod',
    'testing': 'Jafar Test',
    'development': 'Jafar Dev',
    'stress': 'Jafar Stress',
}, fallback=True)

DATE_FORMAT = '%Y-%m-%d'
LOCAL_TIMEZONE = pytz.timezone('Europe/Moscow')

LAUNCHER_TRANSLATIONS_COLLECTION = 'localization.launcher_translations'
LAUNCHER_EXPERIMENTS_COLLECTION = 'localization.launcher'

if DEPLOYMENT_TYPE == QLOUD:
    QLOUD_PREPACKAGED_SNAPSHOT_PATH = os.environ.get('QLOUD_JAFAR_SNAPSHOT')

HARD_BANNED_APPS = {
    'com.yandex.launcher',
    'com.yandex.phone.ui',
    'com.yandex.launcher.xdev'
}

HASOFFERS_API_TOKEN = os.environ.get('HASOFFERS_API_TOKEN')
HASOFFERS_URL = 'https://yandex.api.hasoffers.com/Apiv3/json?NetworkToken={}'

DISLIKED_ITEMS_TTL = yenv.choose_key_by_type({
    'production': 180 * 24 * 60 * 60,
    'testing': 10 * 60,
}, fallback=True)

REMOVED_ITEMS_TTL = yenv.choose_key_by_type({
    'production': 180 * 24 * 60 * 60,
    'testing': 10 * 60,
}, fallback=True)

MDS_ACCESS_TOKEN = os.environ.get('MDS_ACCESS_TOKEN')

# TVM2
TVM_CLIENT_ID = yenv.choose_key_by_type({
    'production': '2001918',
    'testing': '2001914',
    'development': '2001910',
}, fallback=True)
TVM_CLIENT_SECRET = os.environ.get('JAFAR_TVM_CLIENT_SECRET')

BLACKBOX_CLIENT = yenv.choose_key_by_type({
    'production': BlackboxClientId.Prod,
    'testing': BlackboxClientId.Mimino,
}, fallback=True)
BLACKBOX_CLIENT_ID = BLACKBOX_CLIENT.value

BLACKBOX_URL = BLACKBOX_MAP[BLACKBOX_CLIENT]['url']

ZORA_CLIENT_ID = '2000193'
ZORA_URL = 'http://zora.yandex.net:8166/'
ZORA_PROXIES = {'https': ZORA_URL, 'http': ZORA_URL}

DIRECT_CLIENT_ID = yenv.choose_key_by_type({
    'production': '2000390',
    'testing': '2000693'
}, fallback=True)

YPHONE_DEVICE_MODELS = [
    ('Yandex', 'YNDX-000SB'),
]

S3_MDS_ACCESS_KEY_ID = os.environ.get('S3_MDS_ACCESS_KEY_ID')
S3_MDS_ACCESS_SECRET_KEY = os.environ.get('S3_MDS_ACCESS_SECRET_KEY')
S3_MDS_BUCKET_NAME = 'lnchr-files'
S3_MDS_HOST = yenv.choose_key_by_type({
    'production': 's3.mds.yandex.net',
    'testing': 's3.mdst.yandex.net',
}, fallback=True)

S3_URL_TEMPLATE = yenv.choose_key_by_type({
    'production': 'https://lnchr-files.s3.yandex.net/%s',
    'testing': 'https://lnchr-files.s3.mdst.yandex.net/%s',
}, fallback=True)

MDS_HOST = yenv.choose_key_by_type({
    'production': 'https://storage.mds.yandex.net',
    'testing': 'https://storage.mdst.yandex.net',
}, fallback=True)

MDS_URL_TEMPLATE = '{mds_host}/get-lnchr/%s'.format(mds_host=MDS_HOST)

RESIZER_HOST = yenv.choose_key_by_type({
    'production': 'https://resize.yandex.net',
    'testing': 'https://resize.rs.yandex.net',
    'stress': 'https://resize.load.yandex.net',
}, fallback=True)

RESIZER_SECRET_KEY = os.environ.get('RESIZER_SECRET_KEY')
