# encoding: utf-8
import time
from urlparse import urlunparse, ParseResult

from flask import Flask, Markup, render_template, request, redirect, url_for
from flask_admin import Admin
from flask_admin.base import MenuLink
from flask_login import current_user, login_user, LoginManager
from flask_oauthlib.client import OAuth
from os.path import abspath, dirname

import yenv

import localization_admin.log as logging
from db import db
from settings import CONTACTS, BACKGROUND_COLOR, get_qloud_envs_urls
from views.access import get_admins
from views.common import mode_select_view
from views.index import index_view

app = Flask("localization-flask-admin")
app.config.from_object("localization_admin.settings")
app.root_path = abspath(dirname(__file__))

admin = Admin(app, app.config['ADMIN_LOGO'], index_view=index_view, static_url_path='/admin', base_template='base.html')

timeouts = [0.5, 1, 2, 10, 60]
while True:
    try:
        db.init_app(app)
        break
    except Exception as e:
        if len(timeouts):
            logging.error("Unable to start app: db error: " + unicode(e))
            timeout = timeouts.pop(0)
            time.sleep(timeout)
            continue
        else:
            raise

from views.localization import project_localization_view
from views.permissions import permissions_view
from views.support_info import (
    locales_view,
    mobile_operators_view,
    operation_systems_view,
    yandex_apps_view,
    startrek_interaction_view
)

# Settings
admin.add_view(mode_select_view)

# Localizations
admin.add_view(project_localization_view)

# Support Info
admin.add_view(locales_view)
admin.add_view(operation_systems_view)
admin.add_view(mobile_operators_view)
admin.add_view(yandex_apps_view)
admin.add_view(startrek_interaction_view)


# add links
class MenuLinkRuntime(MenuLink):
    def __init__(self, name, env_type, category=None):
        super(MenuLinkRuntime, self).__init__(name, category=category)
        self.env_type = env_type

    def get_url(self):
        environments = get_qloud_envs_urls()
        netloc = environments[self.env_type]
        if not netloc and request:
            netloc = environments[yenv.type] = request.host

        url_parts = ParseResult(scheme=request.scheme,
                                netloc=netloc,
                                path=request.path, params='',
                                query=request.query_string, fragment='')
        return urlunparse(url_parts)


for env_type in get_qloud_envs_urls().keys():
    if env_type != yenv.type:
        name = Markup('go to <b>%s</b>' % env_type)
        admin.add_link(MenuLinkRuntime(name=name, category='Environments', env_type=env_type))

# Permissions
admin.add_view(permissions_view)

oauth = OAuth()
yandex_oauth = oauth.remote_app(
    'yandex-team', app_key='OAUTH_YANDEX'
)
login_manager = LoginManager()
login_manager.init_app(app)


class YandexUser(object):
    def __init__(self, login):
        self.login = login

    is_authenticated = True
    is_active = True
    is_anonymous = False

    def get_id(self):
        return self.login

    @property
    def is_admin(self):
        return self.login in get_admins()


@app.route('/oauth_callback')
def oauth_callback():
    """
    Handles getting token from yandex-team OAuth and user info processing
    """
    if current_user.is_anonymous:
        resp = yandex_oauth.authorized_response()
        token = resp['access_token']
        user_info = yandex_oauth.get(app.config['OAUTH_INFO_URL'], token=[token]).data
        user = YandexUser(login=user_info['login'])
        login_user(user, remember=True)
    return redirect(url_for('admin.index'))


@login_manager.user_loader
def load_user(login):
    return YandexUser(login=login)


# Errors
@app.errorhandler(401)
def unauthorized(e):
    return render_template('errors/401.html', contacts=CONTACTS,
                           body_background_color=BACKGROUND_COLOR), 401  # This type of exception is raised only by yauth


@app.errorhandler(403)
def page_forbidden(e):
    return render_template('errors/403.html', contacts=get_admins(),
                           body_background_color=BACKGROUND_COLOR, exception=e), 403


@app.errorhandler(404)
def page_not_found(e):
    return render_template('errors/404.html', contacts=CONTACTS,
                           body_background_color=BACKGROUND_COLOR), 404


# User authentication
@app.before_request
def before_request():
    request._cookies_to_set = {}
    if current_user.is_anonymous and request.path != '/oauth_callback':
        return yandex_oauth.authorize(callback=url_for('oauth_callback', _external=True, _scheme='https'))
    request.yauser = current_user
    request.admin_mode = request.cookies.get('admin_mode', False) == '1' and request.yauser.is_admin


@app.after_request
def after_request(response):
    for cookie_name, cookie_value in request._cookies_to_set.items():
        response.set_cookie(cookie_name, cookie_value)
    return response


class ReverseProxied(object):
    """
    Preserving scheme and host from initial request
    """

    def __init__(self, app):
        self.app = app

    def __call__(self, environ, start_response):
        scheme = environ.get('HTTP_X_SCHEME') or environ.get('HTTP_X_FORWARDED_PROTO')
        if scheme:
            environ['wsgi.url_scheme'] = scheme
        server = environ.get('HTTP_X_FORWARDED_SERVER')
        if server:
            environ['HTTP_HOST'] = server
        return self.app(environ, start_response)


app.wsgi_app = ReverseProxied(app.wsgi_app)

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=8080)
