# encoding: utf-8
from mongoengine import ValidationError
from localization_admin.db import db
from localization_admin.models.support_info.locales import (
    get_locale_languages,
    get_locale_countries,
    is_valid_language,
    is_valid_country,
    is_valid_locale,
)


def validate_language(value):
    if not is_valid_language(value.lstrip('!')):
        errors = {'language': '"%s" is not valid language id' % value}
        raise ValidationError('', errors=errors)
    return True


def validate_country(value):
    if not is_valid_country(value.lstrip('!')):
        errors = {'country': '"%s" is not valid country id' % value}
        raise ValidationError('', errors=errors)
    return True


class Locale(db.EmbeddedDocument):
    language = db.TaggedStringField(regex='^!?[a-zA-Z]{2}$', help_text='Two lowercase letters: ru or !be',
                                    tags=get_locale_languages, validation=validate_language)
    country = db.TaggedStringField(regex='^!?[a-zA-Z]{2}$', help_text='Two uppercase letters: ru or !be',
                                   tags=get_locale_countries, validation=validate_country)

    def validate(self, clean=True):
        result = super(Locale, self).validate(clean=clean)
        if (self.language and self.country
            and self.language[0] != '!' and self.country[0] != '!'
            and not is_valid_locale(self.language, self.country)):
            raise ValidationError('"%s_%s" is not valid locale' % (self.language, self.country))
        return result
