# coding: utf-8
from mongoengine import ValidationError
from operator import itemgetter

from localization_admin.db import db
from localization_admin.models.details import Locale, TimeRange, DeviceModel, MobilePlatform, OsVersion
from localization_admin.models.details import VersionRangeApplication, LocalizationClid
from localization_admin.models.support_info import MobileOperator
from .extended_params import ExtendedParams

DEVICE_TYPES = ['phone', 'tablet', 'tv']


def validate_clids(value):
    unique_numbers = frozenset(map(itemgetter('number'), value))
    if len(unique_numbers) != len(value):
        raise ValidationError(
            'numbers (%s) must be unique because AND condition applying for clids' % list(unique_numbers)
        )

    for clid in value:
        clid_values = clid.value
        if len(clid_values) == 0:
            raise ValidationError("must be defined at least one clid's value (clid number = %d)" % clid.number)
        elif len(frozenset(clid_values)) != len(clid_values):
            raise ValidationError("eliminate duplicates in clid's value (clid number = %d)" % clid.number)

    return True


def get_targeting_conditions(resource='value', override={}):
    attrs = {
        'meta': {'strict': False},

        'enabled': db.BooleanField(required=True, default=True,
                                   help_text='Is this %s enabled?' % resource),
        'uuids': db.ListField(db.StringField(regex='^[0-9a-f]{32}$',
                                             help_text="Alpha numerical sequence of length 32"),
                              help_text="""If this field is filled, %s
            will be active only for users with ids from this list (generally used for testing)""" % resource,
                              default=None),
        'locale': db.EmbeddedDocumentField(Locale, help_text="""%s will be active only for users with selected
                country/language combination""" % resource.capitalize()),
        'operators': db.ListField(db.ReferenceField(MobileOperator, required=True),
                                  help_text="""%s will be active only for users with one of the
                                                              selected mobile operators""" % resource.capitalize()),
        'region_ids_init': db.GeoCodesListField(verbose_name='Init Region', default=None,
                                                help_text="%s will be active only for users who activate device "
                                                          "from ANY of the selected places" % resource.capitalize()),
        'region_ids_init_blacklist': db.GeoCodesListField(verbose_name='Init Region Blacklist', default=None,
                                                          help_text="%s will NOT be active for users from ANY of "
                                                                    "selected initial places" % resource.capitalize()),
        'region_ids': db.GeoCodesListField(verbose_name='Current Region', default=None,
                                           help_text="%s will be active only for users from ANY of selected places" %
                                                     resource.capitalize()),
        'region_ids_blacklist': db.GeoCodesListField(verbose_name='Current Region Blacklist', default=None,
                                                     help_text="%s will NOT be active for users from ANY of "
                                                               "selected places" % resource.capitalize()),
        'use_ip': db.BooleanField(db_field='detectOperatorByIp', default=False,
                                  help_text="""Use user's ip to determine his mobile
                        operator. Otherwise application data will be used to determine it."""),
        'time': db.EmbeddedDocumentField(TimeRange, help_text="""%s will be active only during selected period of time.
                                                    From <= value < To""" % resource.capitalize()),
        'audience': db.FloatField(min_value=0, max_value=1, help_text="""Percentage of users (from 0 to 1)"""),
        'audience_offset': db.FloatField(min_value=0, max_value=1, help_text="Skip this percentage of users"),
        'deviceTypes': db.ListField(db.StringField(choices=DEVICE_TYPES), verbose_name='Device Types',
                                    help_text="""%s will be active only
                                    for users with with selected types of devices""" % resource.capitalize(),
                                    default=None),
        'models': db.ListField(db.EmbeddedDocumentField(DeviceModel), verbose_name='Device Models',
                               help_text="""%s will be active only for users
                    with one of the selected model/manufacturer combinations""" % resource.capitalize(), default=None),
        'clids': db.ListField(db.EmbeddedDocumentField(LocalizationClid), validation=validate_clids,
                              help_text="Required subset of clids (numbers must be unique). "
                                        "clids will be checked using AND condition"),
        'platforms': db.ListField(db.EmbeddedDocumentField(MobilePlatform), help_text="""%s will be active only for
                    users with one of the selected operation systems""" % resource.capitalize(), default=None),
        'applications': db.ListField(db.EmbeddedDocumentField(VersionRangeApplication), help_text="""%s will be
                    active only if request came through one of the
                    selected applications""" % resource.capitalize(), default=None),
        'os_version': db.EmbeddedDocumentField(OsVersion, default=None),
        'extendedParams': db.EmbeddedDocumentField(ExtendedParams, verbose_name='Extended Parameters',
                                                   help_text="These are rarely using parameters")
    }

    attrs.update(override)

    return type(resource.capitalize() + 'TargetingConditions', (db.EmbeddedDocument,), attrs)


class ValueBlock(db.EmbeddedDocument):
    """
        Elementary part of any localization collection
    """
    value = db.StringField()
    conditions = db.EmbeddedDocumentField(get_targeting_conditions())
