# coding: utf-8
import json
from diff_match_patch import diff_match_patch

from flask import request, Markup
from mongoengine.connection import get_connection
from localization_admin.helpers import generate_id
import localization_admin.settings as settings


def disabled(field_instance):
    """
        Disabled fields are grayed(not editable) and does not submit their values
        Does not work on complex fields
    """
    field_instance.disabled = True
    return field_instance


def default_id():
    if request.args.get('id', None):
        return None
    else:
        return generate_id(lambda *_: True)


def get_localization_projects():
    projects = []
    mongo_db = get_connection(alias='localizations')[settings.LOCALIZATIONS_DBNAME]
    for collection_name in mongo_db.collection_names():
        if not collection_name.startswith(settings.COMMON_LOCALIZATION_PREFIX):
            continue
        project_name = collection_name[len(settings.COMMON_LOCALIZATION_PREFIX):]
        if project_name.startswith('_'):
            continue
        if project_name in settings.EXCLUDED_LOCALIZATION_PROJECTS:
            continue
        projects.append(project_name)

    return projects


def to_dict(model):
    return model.to_mongo().to_dict() if model else None


def prettify_json(json_data, indent=2):
    return json.dumps(json.loads(json_data), indent=indent, encoding='utf-8', ensure_ascii=False)


OP_TEMPLATE = {
    diff_match_patch.DIFF_DELETE: u"<del style=\"background:#fdafaf;\">%s</del>",
    diff_match_patch.DIFF_INSERT: u"<ins style=\"background:#abffab;\">%s</ins>",
    diff_match_patch.DIFF_EQUAL: u"<span>%s</span>",
}


def format_diff(document_before, document_after):
    def htmlize(op, data):
        lines = []
        for line in data.split('\n'):
            line = line.replace("&", "&amp;").replace("<", "&lt;").replace(">", "&gt;")
            indent = len(line) - len(line.lstrip())
            line = u'&nbsp;' * indent + (OP_TEMPLATE[op] % line.strip())
            lines.append(line)
        return u'<br>'.join(lines)

    if document_before and document_after:
        before = prettify_json(document_before)
        after = prettify_json(document_after)
        dmp = diff_match_patch()
        diffs = dmp.diff_main(before, after)
        dmp.diff_cleanupSemantic(diffs)
        html = [htmlize(op, data) for op, data in diffs]
        return Markup(u''.join(html))
    elif document_after:
        text = htmlize(prettify_json(document_after))
        return Markup(htmlize(diff_match_patch.DIFF_INSERT, text))
    elif document_before:
        text = htmlize(prettify_json(document_before))
        return Markup(htmlize(diff_match_patch.DIFF_DELETE, text))
