# encoding: utf-8

import re
from operator import itemgetter
from functools import partial
from itertools import groupby
from mongoengine.connection import ConnectionError
from localization_admin.models.support_info.locale import LocaleModel

re_locale = re.compile(u"""
    (?P<language>.{2})              # language code, ru
    _                               # just code separator
    (?P<country>.{2})               # country code, US
    ;                               # locale & descriptions separator
    (?P<language_description>.+?)   # language description
    [ ]\(
    (?P<country_description>.+)     # country description in parentheses
    \)
""", re.U | re.X)


def initialize():
    try:
        if LocaleModel.objects.count() != 0:
            return
    except ConnectionError:
        return

    LocaleModel.objects.delete()  # remove all
    get_values = itemgetter('language', 'language_description', 'country', 'country_description')
    for locale in get_valid_locales():
        groups = re_locale.match(';'.join(locale)).groupdict()
        doc = LocaleModel(*get_values(groups))
        doc.save()


def is_valid_language(lang_id):
    return lang_id in get_language_ids()


def is_valid_country(country_id):
    return country_id in get_country_ids()


def is_valid_locale(language, country):
    return LocaleModel.objects(language=language, country=country).first() is not None


def default_format(data):
    assert len(data) == 2, 'expected tuple with values: code, description'
    return u'{0},{0} ({1})'.format(*data)


def get_locale_countries(format=default_format):
    initialize()
    mapped = {}
    for key, group in groupby(LocaleModel.objects().order_by('country'), itemgetter('country')):
        doc = next(group)
        mapped[key] = doc['country_title']
    return [format(v) if format else v for v in dict(mapped).iteritems()]


def get_valid_locales():
    return filter(lambda x: len(x[0]) == 5, locales.iteritems())


def get_locale_languages(format=default_format):
    initialize()
    mapped = {}
    for key, group in groupby(LocaleModel.objects().order_by('language'), itemgetter('language')):
        doc = next(group)
        mapped[doc['language']] = doc['language_title']
    return [format(v) if format else v for v in dict(mapped).iteritems()]


def get_ids(get_all_func):
    return frozenset(map(itemgetter(0), get_all_func(None)))


get_language_ids = partial(get_ids, get_locale_languages)
get_country_ids = partial(get_ids, get_locale_countries)

# source: https://github.com/umpirsky/locale-list/tree/master/data/ru_RU
locales = {
    u'af': u'африкаанс',
    u'af_NA': u'африкаанс (Намибия)',
    u'af_ZA': u'африкаанс (ЮАР)',
    u'ak': u'акан',
    u'ak_GH': u'акан (Гана)',
    u'am': u'амхарский',
    u'am_ET': u'амхарский (Эфиопия)',
    u'ar': u'арабский',
    u'ar_AE': u'арабский (ОАЭ)',
    u'ar_BH': u'арабский (Бахрейн)',
    u'ar_DJ': u'арабский (Джибути)',
    u'ar_DZ': u'арабский (Алжир)',
    u'ar_EG': u'арабский (Египет)',
    u'ar_EH': u'арабский (Западная Сахара)',
    u'ar_ER': u'арабский (Эритрея)',
    u'ar_IL': u'арабский (Израиль)',
    u'ar_IQ': u'арабский (Ирак)',
    u'ar_JO': u'арабский (Иордания)',
    u'ar_KM': u'арабский (Коморские о-ва)',
    u'ar_KW': u'арабский (Кувейт)',
    u'ar_LB': u'арабский (Ливан)',
    u'ar_LY': u'арабский (Ливия)',
    u'ar_MA': u'арабский (Марокко)',
    u'ar_MR': u'арабский (Мавритания)',
    u'ar_OM': u'арабский (Оман)',
    u'ar_PS': u'арабский (Палестинские территории)',
    u'ar_QA': u'арабский (Катар)',
    u'ar_SA': u'арабский (Саудовская Аравия)',
    u'ar_SD': u'арабский (Судан)',
    u'ar_SO': u'арабский (Сомали)',
    u'ar_SS': u'арабский (Южный Судан)',
    u'ar_SY': u'арабский (Сирия)',
    u'ar_TD': u'арабский (Чад)',
    u'ar_TN': u'арабский (Тунис)',
    u'ar_YE': u'арабский (Йемен)',
    u'as': u'ассамский',
    u'as_IN': u'ассамский (Индия)',
    u'az': u'азербайджанский',
    u'az_AZ': u'азербайджанский (Азербайджан)',
    u'az_Cyrl': u'азербайджанский (кириллица)',
    u'az_Cyrl_AZ': u'азербайджанский (кириллица, Азербайджан)',
    u'az_Latn': u'азербайджанский (латиница)',
    u'az_Latn_AZ': u'азербайджанский (латиница, Азербайджан)',
    u'be': u'белорусский',
    u'be_BY': u'белорусский (Беларусь)',
    u'bg': u'болгарский',
    u'bg_BG': u'болгарский (Болгария)',
    u'bm': u'бамбарийский',
    u'bm_Latn': u'бамбарийский (латиница)',
    u'bm_Latn_ML': u'бамбарийский (латиница, Мали)',
    u'bn': u'бенгальский',
    u'bn_BD': u'бенгальский (Бангладеш)',
    u'bn_IN': u'бенгальский (Индия)',
    u'bo': u'тибетский',
    u'bo_CN': u'тибетский (Китай)',
    u'bo_IN': u'тибетский (Индия)',
    u'br': u'бретонский',
    u'br_FR': u'бретонский (Франция)',
    u'bs': u'боснийский',
    u'bs_BA': u'боснийский (Босния и Герцеговина)',
    u'bs_Cyrl': u'боснийский (кириллица)',
    u'bs_Cyrl_BA': u'боснийский (кириллица, Босния и Герцеговина)',
    u'bs_Latn': u'боснийский (латиница)',
    u'bs_Latn_BA': u'боснийский (латиница, Босния и Герцеговина)',
    u'ca': u'каталанский',
    u'ca_AD': u'каталанский (Андорра)',
    u'ca_ES': u'каталанский (Испания)',
    u'ca_FR': u'каталанский (Франция)',
    u'ca_IT': u'каталанский (Италия)',
    u'cs': u'чешский',
    u'cs_CZ': u'чешский (Чехия)',
    u'cy': u'валлийский',
    u'cy_GB': u'валлийский (Великобритания)',
    u'da': u'датский',
    u'da_DK': u'датский (Дания)',
    u'da_GL': u'датский (Гренландия)',
    u'de': u'немецкий',
    u'de_AT': u'немецкий (Австрия)',
    u'de_BE': u'немецкий (Бельгия)',
    u'de_CH': u'немецкий (Швейцария)',
    u'de_DE': u'немецкий (Германия)',
    u'de_LI': u'немецкий (Лихтенштейн)',
    u'de_LU': u'немецкий (Люксембург)',
    u'dz': u'дзонг-кэ',
    u'dz_BT': u'дзонг-кэ (Бутан)',
    u'ee': u'эве',
    u'ee_GH': u'эве (Гана)',
    u'ee_TG': u'эве (Того)',
    u'el': u'греческий',
    u'el_CY': u'греческий (Кипр)',
    u'el_GR': u'греческий (Греция)',
    u'en': u'английский',
    u'en_AG': u'английский (Антигуа и Барбуда)',
    u'en_AI': u'английский (Ангилья)',
    u'en_AS': u'английский (Американское Самоа)',
    u'en_AU': u'английский (Австралия)',
    u'en_BB': u'английский (Барбадос)',
    u'en_BE': u'английский (Бельгия)',
    u'en_BM': u'английский (Бермудские о-ва)',
    u'en_BS': u'английский (Багамские о-ва)',
    u'en_BW': u'английский (Ботсвана)',
    u'en_BZ': u'английский (Белиз)',
    u'en_CA': u'английский (Канада)',
    u'en_CC': u'английский (Кокосовые о-ва)',
    u'en_CK': u'английский (о-ва Кука)',
    u'en_CM': u'английский (Камерун)',
    u'en_CX': u'английский (о-в Рождества)',
    u'en_DG': u'английский (Диего-Гарсия)',
    u'en_DM': u'английский (Доминика)',
    u'en_ER': u'английский (Эритрея)',
    u'en_FJ': u'английский (Фиджи)',
    u'en_FK': u'английский (Фолклендские о-ва)',
    u'en_FM': u'английский (Федеративные Штаты Микронезии)',
    u'en_GB': u'английский (Великобритания)',
    u'en_GD': u'английский (Гренада)',
    u'en_GG': u'английский (Гернси)',
    u'en_GH': u'английский (Гана)',
    u'en_GI': u'английский (Гибралтар)',
    u'en_GM': u'английский (Гамбия)',
    u'en_GU': u'английский (Гуам)',
    u'en_GY': u'английский (Гайана)',
    u'en_HK': u'английский (Гонконг (особый район))',
    u'en_IE': u'английский (Ирландия)',
    u'en_IM': u'английский (О-в Мэн)',
    u'en_IN': u'английский (Индия)',
    u'en_IO': u'английский (Британская территория в Индийском океане)',
    u'en_JE': u'английский (Джерси)',
    u'en_JM': u'английский (Ямайка)',
    u'en_KE': u'английский (Кения)',
    u'en_KI': u'английский (Кирибати)',
    u'en_KN': u'английский (Сент-Китс и Невис)',
    u'en_KY': u'английский (Каймановы о-ва)',
    u'en_LC': u'английский (Сент-Люсия)',
    u'en_LR': u'английский (Либерия)',
    u'en_LS': u'английский (Лесото)',
    u'en_MG': u'английский (Мадагаскар)',
    u'en_MH': u'английский (Маршалловы о-ва)',
    u'en_MO': u'английский (Макао (особый район))',
    u'en_MP': u'английский (Северные Марианские о-ва)',
    u'en_MS': u'английский (Монтсеррат)',
    u'en_MT': u'английский (Мальта)',
    u'en_MU': u'английский (Маврикий)',
    u'en_MW': u'английский (Малави)',
    u'en_MY': u'английский (Малайзия)',
    u'en_NA': u'английский (Намибия)',
    u'en_NF': u'английский (о-в Норфолк)',
    u'en_NG': u'английский (Нигерия)',
    u'en_NR': u'английский (Науру)',
    u'en_NU': u'английский (Ниуэ)',
    u'en_NZ': u'английский (Новая Зеландия)',
    u'en_PG': u'английский (Папуа – Новая Гвинея)',
    u'en_PH': u'английский (Филиппины)',
    u'en_PK': u'английский (Пакистан)',
    u'en_PN': u'английский (Питкэрн)',
    u'en_PR': u'английский (Пуэрто-Рико)',
    u'en_PW': u'английский (Палау)',
    u'en_RW': u'английский (Руанда)',
    u'en_SB': u'английский (Соломоновы о-ва)',
    u'en_SC': u'английский (Сейшельские о-ва)',
    u'en_SD': u'английский (Судан)',
    u'en_SG': u'английский (Сингапур)',
    u'en_SH': u'английский (О-в Св. Елены)',
    u'en_SL': u'английский (Сьерра-Леоне)',
    u'en_SS': u'английский (Южный Судан)',
    u'en_SX': u'английский (Синт-Мартен)',
    u'en_SZ': u'английский (Свазиленд)',
    u'en_TC': u'английский (О-ва Тёркс и Кайкос)',
    u'en_TK': u'английский (Токелау)',
    u'en_TO': u'английский (Тонга)',
    u'en_TT': u'английский (Тринидад и Тобаго)',
    u'en_TV': u'английский (Тувалу)',
    u'en_TZ': u'английский (Танзания)',
    u'en_UG': u'английский (Уганда)',
    u'en_UM': u'английский (Внешние малые о-ва (США))',
    u'en_US': u'английский (Соединенные Штаты)',
    u'en_VC': u'английский (Сент-Винсент и Гренадины)',
    u'en_VG': u'английский (Виргинские о-ва (Британские))',
    u'en_VI': u'английский (Виргинские о-ва (США))',
    u'en_VU': u'английский (Вануату)',
    u'en_WS': u'английский (Самоа)',
    u'en_ZA': u'английский (ЮАР)',
    u'en_ZM': u'английский (Замбия)',
    u'en_ZW': u'английский (Зимбабве)',
    u'eo': u'эсперанто',
    u'es': u'испанский',
    u'es_AR': u'испанский (Аргентина)',
    u'es_BO': u'испанский (Боливия)',
    u'es_CL': u'испанский (Чили)',
    u'es_CO': u'испанский (Колумбия)',
    u'es_CR': u'испанский (Коста-Рика)',
    u'es_CU': u'испанский (Куба)',
    u'es_DO': u'испанский (Доминиканская Республика)',
    u'es_EA': u'испанский (Сеута и Мелилья)',
    u'es_EC': u'испанский (Эквадор)',
    u'es_ES': u'испанский (Испания)',
    u'es_GQ': u'испанский (Экваториальная Гвинея)',
    u'es_GT': u'испанский (Гватемала)',
    u'es_HN': u'испанский (Гондурас)',
    u'es_IC': u'испанский (Канарские о-ва)',
    u'es_MX': u'испанский (Мексика)',
    u'es_NI': u'испанский (Никарагуа)',
    u'es_PA': u'испанский (Панама)',
    u'es_PE': u'испанский (Перу)',
    u'es_PH': u'испанский (Филиппины)',
    u'es_PR': u'испанский (Пуэрто-Рико)',
    u'es_PY': u'испанский (Парагвай)',
    u'es_SV': u'испанский (Сальвадор)',
    u'es_US': u'испанский (Соединенные Штаты)',
    u'es_UY': u'испанский (Уругвай)',
    u'es_VE': u'испанский (Венесуэла)',
    u'et': u'эстонский',
    u'et_EE': u'эстонский (Эстония)',
    u'eu': u'баскский',
    u'eu_ES': u'баскский (Испания)',
    u'fa': u'персидский',
    u'fa_AF': u'персидский (Афганистан)',
    u'fa_IR': u'персидский (Иран)',
    u'ff': u'фулах',
    u'ff_CM': u'фулах (Камерун)',
    u'ff_GN': u'фулах (Гвинея)',
    u'ff_MR': u'фулах (Мавритания)',
    u'ff_SN': u'фулах (Сенегал)',
    u'fi': u'финский',
    u'fi_FI': u'финский (Финляндия)',
    u'fo': u'фарерский',
    u'fo_FO': u'фарерский (Фарерские о-ва)',
    u'fr': u'французский',
    u'fr_BE': u'французский (Бельгия)',
    u'fr_BF': u'французский (Буркина-Фасо)',
    u'fr_BI': u'французский (Бурунди)',
    u'fr_BJ': u'французский (Бенин)',
    u'fr_BL': u'французский (Сен-Бартельми)',
    u'fr_CA': u'французский (Канада)',
    u'fr_CD': u'французский (Конго - Киншаса)',
    u'fr_CF': u'французский (ЦАР)',
    u'fr_CG': u'французский (Конго - Браззавиль)',
    u'fr_CH': u'французский (Швейцария)',
    u'fr_CI': u'французский (Кот-д’Ивуар)',
    u'fr_CM': u'французский (Камерун)',
    u'fr_DJ': u'французский (Джибути)',
    u'fr_DZ': u'французский (Алжир)',
    u'fr_FR': u'французский (Франция)',
    u'fr_GA': u'французский (Габон)',
    u'fr_GF': u'французский (Французская Гвиана)',
    u'fr_GN': u'французский (Гвинея)',
    u'fr_GP': u'французский (Гваделупа)',
    u'fr_GQ': u'французский (Экваториальная Гвинея)',
    u'fr_HT': u'французский (Гаити)',
    u'fr_KM': u'французский (Коморские о-ва)',
    u'fr_LU': u'французский (Люксембург)',
    u'fr_MA': u'французский (Марокко)',
    u'fr_MC': u'французский (Монако)',
    u'fr_MF': u'французский (Сен-Мартен)',
    u'fr_MG': u'французский (Мадагаскар)',
    u'fr_ML': u'французский (Мали)',
    u'fr_MQ': u'французский (Мартиника)',
    u'fr_MR': u'французский (Мавритания)',
    u'fr_MU': u'французский (Маврикий)',
    u'fr_NC': u'французский (Новая Каледония)',
    u'fr_NE': u'французский (Нигер)',
    u'fr_PF': u'французский (Французская Полинезия)',
    u'fr_PM': u'французский (Сен-Пьер и Микелон)',
    u'fr_RE': u'французский (Реюньон)',
    u'fr_RW': u'французский (Руанда)',
    u'fr_SC': u'французский (Сейшельские о-ва)',
    u'fr_SN': u'французский (Сенегал)',
    u'fr_SY': u'французский (Сирия)',
    u'fr_TD': u'французский (Чад)',
    u'fr_TG': u'французский (Того)',
    u'fr_TN': u'французский (Тунис)',
    u'fr_VU': u'французский (Вануату)',
    u'fr_WF': u'французский (Уоллис и Футуна)',
    u'fr_YT': u'французский (Майотта)',
    u'fy': u'западно-фризский',
    u'fy_NL': u'западно-фризский (Нидерланды)',
    u'ga': u'ирландский',
    u'ga_IE': u'ирландский (Ирландия)',
    u'gd': u'гэльский',
    u'gd_GB': u'гэльский (Великобритания)',
    u'gl': u'галисийский',
    u'gl_ES': u'галисийский (Испания)',
    u'gu': u'гуджарати',
    u'gu_IN': u'гуджарати (Индия)',
    u'gv': u'мэнский',
    u'gv_IM': u'мэнский (О-в Мэн)',
    u'ha': u'хауса',
    u'ha_GH': u'хауса (Гана)',
    u'ha_Latn': u'хауса (латиница)',
    u'ha_Latn_GH': u'хауса (латиница, Гана)',
    u'ha_Latn_NE': u'хауса (латиница, Нигер)',
    u'ha_Latn_NG': u'хауса (латиница, Нигерия)',
    u'ha_NE': u'хауса (Нигер)',
    u'ha_NG': u'хауса (Нигерия)',
    u'he': u'иврит',
    u'he_IL': u'иврит (Израиль)',
    u'hi': u'хинди',
    u'hi_IN': u'хинди (Индия)',
    u'hr': u'хорватский',
    u'hr_BA': u'хорватский (Босния и Герцеговина)',
    u'hr_HR': u'хорватский (Хорватия)',
    u'hu': u'венгерский',
    u'hu_HU': u'венгерский (Венгрия)',
    u'hy': u'армянский',
    u'hy_AM': u'армянский (Армения)',
    u'id': u'индонезийский',
    u'id_ID': u'индонезийский (Индонезия)',
    u'ig': u'игбо',
    u'ig_NG': u'игбо (Нигерия)',
    u'ii': u'сычуань',
    u'ii_CN': u'сычуань (Китай)',
    u'is': u'исландский',
    u'is_IS': u'исландский (Исландия)',
    u'it': u'итальянский',
    u'it_CH': u'итальянский (Швейцария)',
    u'it_IT': u'итальянский (Италия)',
    u'it_SM': u'итальянский (Сан-Марино)',
    u'ja': u'японский',
    u'ja_JP': u'японский (Япония)',
    u'ka': u'грузинский',
    u'ka_GE': u'грузинский (Грузия)',
    u'ki': u'кикуйю',
    u'ki_KE': u'кикуйю (Кения)',
    u'kk': u'казахский',
    u'kk_Cyrl': u'казахский (кириллица)',
    u'kk_Cyrl_KZ': u'казахский (кириллица, Казахстан)',
    u'kk_KZ': u'казахский (Казахстан)',
    u'kl': u'гренландский',
    u'kl_GL': u'гренландский (Гренландия)',
    u'km': u'кхмерский',
    u'km_KH': u'кхмерский (Камбоджа)',
    u'kn': u'каннада',
    u'kn_IN': u'каннада (Индия)',
    u'ko': u'корейский',
    u'ko_KP': u'корейский (КНДР)',
    u'ko_KR': u'корейский (Республика Корея)',
    u'ks': u'кашмири',
    u'ks_Arab': u'кашмири (арабица)',
    u'ks_Arab_IN': u'кашмири (арабица, Индия)',
    u'ks_IN': u'кашмири (Индия)',
    u'kw': u'корнийский',
    u'kw_GB': u'корнийский (Великобритания)',
    u'ky': u'киргизский',
    u'ky_Cyrl': u'киргизский (кириллица)',
    u'ky_Cyrl_KG': u'киргизский (кириллица, Киргизия)',
    u'ky_KG': u'киргизский (Киргизия)',
    u'lb': u'люксембургский',
    u'lb_LU': u'люксембургский (Люксембург)',
    u'lg': u'ганда',
    u'lg_UG': u'ганда (Уганда)',
    u'ln': u'лингала',
    u'ln_AO': u'лингала (Ангола)',
    u'ln_CD': u'лингала (Конго - Киншаса)',
    u'ln_CF': u'лингала (ЦАР)',
    u'ln_CG': u'лингала (Конго - Браззавиль)',
    u'lo': u'лаосский',
    u'lo_LA': u'лаосский (Лаос)',
    u'lt': u'литовский',
    u'lt_LT': u'литовский (Литва)',
    u'lu': u'луба-катанга',
    u'lu_CD': u'луба-катанга (Конго - Киншаса)',
    u'lv': u'латышский',
    u'lv_LV': u'латышский (Латвия)',
    u'mg': u'малагасийский',
    u'mg_MG': u'малагасийский (Мадагаскар)',
    u'mk': u'македонский',
    u'mk_MK': u'македонский (Македония)',
    u'ml': u'малаялам',
    u'ml_IN': u'малаялам (Индия)',
    u'mn': u'монгольский',
    u'mn_Cyrl': u'монгольский (кириллица)',
    u'mn_Cyrl_MN': u'монгольский (кириллица, Монголия)',
    u'mn_MN': u'монгольский (Монголия)',
    u'mr': u'маратхи',
    u'mr_IN': u'маратхи (Индия)',
    u'ms': u'малайский',
    u'ms_BN': u'малайский (Бруней-Даруссалам)',
    u'ms_Latn': u'малайский (латиница)',
    u'ms_Latn_BN': u'малайский (латиница, Бруней-Даруссалам)',
    u'ms_Latn_MY': u'малайский (латиница, Малайзия)',
    u'ms_Latn_SG': u'малайский (латиница, Сингапур)',
    u'ms_MY': u'малайский (Малайзия)',
    u'ms_SG': u'малайский (Сингапур)',
    u'mt': u'мальтийский',
    u'mt_MT': u'мальтийский (Мальта)',
    u'my': u'бирманский',
    u'my_MM': u'бирманский (Мьянма (Бирма))',
    u'nb': u'норвежский букмол',
    u'nb_NO': u'норвежский букмол (Норвегия)',
    u'nb_SJ': u'норвежский букмол (Шпицберген и Ян-Майен)',
    u'nd': u'северный ндебели',
    u'nd_ZW': u'северный ндебели (Зимбабве)',
    u'ne': u'непальский',
    u'ne_IN': u'непальский (Индия)',
    u'ne_NP': u'непальский (Непал)',
    u'nl': u'голландский',
    u'nl_AW': u'голландский (Аруба)',
    u'nl_BE': u'голландский (Бельгия)',
    u'nl_BQ': u'голландский (Бонэйр, Синт-Эстатиус и Саба)',
    u'nl_CW': u'голландский (Кюрасао)',
    u'nl_NL': u'голландский (Нидерланды)',
    u'nl_SR': u'голландский (Суринам)',
    u'nl_SX': u'голландский (Синт-Мартен)',
    u'nn': u'норвежский нюнорск',
    u'nn_NO': u'норвежский нюнорск (Норвегия)',
    u'no': u'норвежский',
    u'no_NO': u'норвежский (Норвегия)',
    u'om': u'оромо',
    u'om_ET': u'оромо (Эфиопия)',
    u'om_KE': u'оромо (Кения)',
    u'or': u'ория',
    u'or_IN': u'ория (Индия)',
    u'os': u'осетинский',
    u'os_GE': u'осетинский (Грузия)',
    u'os_RU': u'осетинский (Россия)',
    u'pa': u'панджаби',
    u'pa_Arab': u'панджаби (арабица)',
    u'pa_Arab_PK': u'панджаби (арабица, Пакистан)',
    u'pa_Guru': u'панджаби (гурмукхи)',
    u'pa_Guru_IN': u'панджаби (гурмукхи, Индия)',
    u'pa_IN': u'панджаби (Индия)',
    u'pa_PK': u'панджаби (Пакистан)',
    u'pl': u'польский',
    u'pl_PL': u'польский (Польша)',
    u'ps': u'пушту',
    u'ps_AF': u'пушту (Афганистан)',
    u'pt': u'португальский',
    u'pt_AO': u'португальский (Ангола)',
    u'pt_BR': u'португальский (Бразилия)',
    u'pt_CV': u'португальский (Кабо-Верде)',
    u'pt_GW': u'португальский (Гвинея-Бисау)',
    u'pt_MO': u'португальский (Макао (особый район))',
    u'pt_MZ': u'португальский (Мозамбик)',
    u'pt_PT': u'португальский (Португалия)',
    u'pt_ST': u'португальский (Сан-Томе и Принсипи)',
    u'pt_TL': u'португальский (Восточный Тимор)',
    u'qu': u'кечуа',
    u'qu_BO': u'кечуа (Боливия)',
    u'qu_EC': u'кечуа (Эквадор)',
    u'qu_PE': u'кечуа (Перу)',
    u'rm': u'романшский',
    u'rm_CH': u'романшский (Швейцария)',
    u'rn': u'рунди',
    u'rn_BI': u'рунди (Бурунди)',
    u'ro': u'румынский',
    u'ro_MD': u'румынский (Молдова)',
    u'ro_RO': u'румынский (Румыния)',
    u'ru': u'русский',
    u'ru_BY': u'русский (Беларусь)',
    u'ru_KG': u'русский (Киргизия)',
    u'ru_KZ': u'русский (Казахстан)',
    u'ru_MD': u'русский (Молдова)',
    u'ru_RU': u'русский (Россия)',
    u'ru_UA': u'русский (Украина)',
    u'rw': u'киньяруанда',
    u'rw_RW': u'киньяруанда (Руанда)',
    u'se': u'северносаамский',
    u'se_FI': u'северносаамский (Финляндия)',
    u'se_NO': u'северносаамский (Норвегия)',
    u'se_SE': u'северносаамский (Швеция)',
    u'sg': u'санго',
    u'sg_CF': u'санго (ЦАР)',
    u'sh': u'сербскохорватский',
    u'sh_BA': u'сербскохорватский (Босния и Герцеговина)',
    u'si': u'сингальский',
    u'si_LK': u'сингальский (Шри-Ланка)',
    u'sk': u'словацкий',
    u'sk_SK': u'словацкий (Словакия)',
    u'sl': u'словенский',
    u'sl_SI': u'словенский (Словения)',
    u'sn': u'шона',
    u'sn_ZW': u'шона (Зимбабве)',
    u'so': u'сомали',
    u'so_DJ': u'сомали (Джибути)',
    u'so_ET': u'сомали (Эфиопия)',
    u'so_KE': u'сомали (Кения)',
    u'so_SO': u'сомали (Сомали)',
    u'sq': u'албанский',
    u'sq_AL': u'албанский (Албания)',
    u'sq_MK': u'албанский (Македония)',
    u'sq_XK': u'албанский (Косово)',
    u'sr': u'сербский',
    u'sr_BA': u'сербский (Босния и Герцеговина)',
    u'sr_Cyrl': u'сербский (кириллица)',
    u'sr_Cyrl_BA': u'сербский (кириллица, Босния и Герцеговина)',
    u'sr_Cyrl_ME': u'сербский (кириллица, Черногория)',
    u'sr_Cyrl_RS': u'сербский (кириллица, Сербия)',
    u'sr_Cyrl_XK': u'сербский (кириллица, Косово)',
    u'sr_Latn': u'сербский (латиница)',
    u'sr_Latn_BA': u'сербский (латиница, Босния и Герцеговина)',
    u'sr_Latn_ME': u'сербский (латиница, Черногория)',
    u'sr_Latn_RS': u'сербский (латиница, Сербия)',
    u'sr_Latn_XK': u'сербский (латиница, Косово)',
    u'sr_ME': u'сербский (Черногория)',
    u'sr_RS': u'сербский (Сербия)',
    u'sr_XK': u'сербский (Косово)',
    u'sv': u'шведский',
    u'sv_AX': u'шведский (Аландские о-ва)',
    u'sv_FI': u'шведский (Финляндия)',
    u'sv_SE': u'шведский (Швеция)',
    u'sw': u'суахили',
    u'sw_KE': u'суахили (Кения)',
    u'sw_TZ': u'суахили (Танзания)',
    u'sw_UG': u'суахили (Уганда)',
    u'ta': u'тамильский',
    u'ta_IN': u'тамильский (Индия)',
    u'ta_LK': u'тамильский (Шри-Ланка)',
    u'ta_MY': u'тамильский (Малайзия)',
    u'ta_SG': u'тамильский (Сингапур)',
    u'te': u'телугу',
    u'te_IN': u'телугу (Индия)',
    u'th': u'тайский',
    u'th_TH': u'тайский (Таиланд)',
    u'ti': u'тигринья',
    u'ti_ER': u'тигринья (Эритрея)',
    u'ti_ET': u'тигринья (Эфиопия)',
    u'tl': u'тагалог',
    u'tl_PH': u'тагалог (Филиппины)',
    u'to': u'тонганский',
    u'to_TO': u'тонганский (Тонга)',
    u'tr': u'турецкий',
    u'tr_CY': u'турецкий (Кипр)',
    u'tr_TR': u'турецкий (Турция)',
    u'ug': u'уйгурский',
    u'ug_Arab': u'уйгурский (арабица)',
    u'ug_Arab_CN': u'уйгурский (арабица, Китай)',
    u'ug_CN': u'уйгурский (Китай)',
    u'uk': u'украинский',
    u'uk_UA': u'украинский (Украина)',
    u'ur': u'урду',
    u'ur_IN': u'урду (Индия)',
    u'ur_PK': u'урду (Пакистан)',
    u'uz': u'узбекский',
    u'uz_AF': u'узбекский (Афганистан)',
    u'uz_Arab': u'узбекский (арабица)',
    u'uz_Arab_AF': u'узбекский (арабица, Афганистан)',
    u'uz_Cyrl': u'узбекский (кириллица)',
    u'uz_Cyrl_UZ': u'узбекский (кириллица, Узбекистан)',
    u'uz_Latn': u'узбекский (латиница)',
    u'uz_Latn_UZ': u'узбекский (латиница, Узбекистан)',
    u'uz_UZ': u'узбекский (Узбекистан)',
    u'vi': u'вьетнамский',
    u'vi_VN': u'вьетнамский (Вьетнам)',
    u'yi': u'идиш',
    u'yo': u'йоруба',
    u'yo_BJ': u'йоруба (Бенин)',
    u'yo_NG': u'йоруба (Нигерия)',
    u'zh': u'китайский',
    u'zh_CN': u'китайский (Китай)',
    u'zh_HK': u'китайский (Гонконг (особый район))',
    u'zh_Hans': u'китайский (упрощенная китайская)',
    u'zh_Hans_CN': u'китайский (упрощенная китайская, Китай)',
    u'zh_Hans_HK': u'китайский (упрощенная китайская, Гонконг (особый район))',
    u'zh_Hans_MO': u'китайский (упрощенная китайская, Макао (особый район))',
    u'zh_Hans_SG': u'китайский (упрощенная китайская, Сингапур)',
    u'zh_Hant': u'китайский (традиционная китайская)',
    u'zh_Hant_HK': u'китайский (традиционная китайская, Гонконг (особый район))',
    u'zh_Hant_MO': u'китайский (традиционная китайская, Макао (особый район))',
    u'zh_Hant_TW': u'китайский (традиционная китайская, Тайвань)',
    u'zh_MO': u'китайский (Макао (особый район))',
    u'zh_SG': u'китайский (Сингапур)',
    u'zh_TW': u'китайский (Тайвань)',
    u'zu': u'зулу',
    u'zu_ZA': u'зулу (ЮАР)',
    ### additional countries by Yandex geobase
    u'tg_TJ': u'таджикский (Таджикистан)',  # 209
    u'tk_TM': u'туркменский (Туркмения)',  # 170
    u'en_GS': u'английский (Южная Георгия и Южные Сандвичевы острова)',  # 101523
    u'fr_TF': u'французский (Французские Южные и Антарктические территории)',  # 120751
    u'dv_MV': u'мальдивский (Мальдивы)',  # 10098
    u'it_VA': u'итальянский (Ватикан)',  # 21359
    u'la_VA': u'латинский (Ватикан)',  # 21359
    u'no_BV': u'норвежский (Остров Буве)',  # 120752
    u'os_OS': u'осетинский (Южная Осетия)',  # 29387
    u'ru_OS': u'русский (Южная Осетия)',  # 29387
    u'en_HM': u'английский (Остров Херд и острова Макдональд)',  # 120750
    u'ab_AB': u'абхазский (Абхазия)',  # 29386
    u'ru_AB': u'русский (Абхазия)',  # 29386
}
