# -*- coding: utf8 -*-
import yenv
import jinja2
from flask import request
from pprint import pformat
from datetime import datetime
from operator import itemgetter

from flask import current_app as app

import secrets
import localization_admin.log as logging
from startrek_client import Startrek, exceptions
from settings import (
    DEFAULT_FOLLOWERS_LIST,
    STARTREK_QUEUE,
    STARTREK_ENABLED,
    STARTREK_ISSUE_TYPE,
)

logger = logging.getLogger(__name__)

client = Startrek(useragent='localization-admin.phone.yandex.net', token=secrets.STARTREK_TOKEN)


def get_summary_template():
    return SUMMARY


def get_description_template():
    return DESCRIPTION


def get_comment_template():
    return COMMENT


def get_queue():
    return STARTREK_QUEUE


def get_ticket_author():
    return dict(login=request.yauser.login)


def get_ticket_assignee():
    return dict(login=request.yauser.login)


def get_ticket_followers():
    return [dict(login=follower) for follower in DEFAULT_FOLLOWERS_LIST]


def formatter(obj):
    if isinstance(obj, (list, set)):
        arr = []
        for v in obj:
            arr.append(formatter(v))
        return arr
    elif isinstance(obj, dict):
        d = {}
        for k, v in obj.items():
            if k == 'user_creation_date':
                d[k] = datetime.fromtimestamp(int(v)).isoformat(sep=' ')
            else:
                d[k] = formatter(v)
        return d
    elif isinstance(obj, datetime):
        return obj.isoformat(sep=' ')
    return obj


def prepare_params(params):
    if 'updated_at' in params:
        del params['updated_at']
    try:
        params = formatter(params)
    except Exception:
        logger.exception("could not format parameters")
    return params


filter_empty_defaults = {
    'enabled': True,
    'operators': [],
    'detectOperatorByIp': False,
    'init_countries': [],
    'clids': [],
}


def filter_empty_fields(obj):
    if obj:
        for k, v in filter_empty_defaults.items():
            if k in obj and obj[k] == v:
                del obj[k]
    return obj


FORMAT_RESTRICTIONS = u'%%%%(code nohighlight)%s%%%%'


def show_restrictions(restrictions):
    for k, v in restrictions.items():
        if k == 'clids':
            restrictions[k] = sorted(map(itemgetter('number', 'value'), v))
            # add here other simplifying code
    if restrictions:
        return FORMAT_RESTRICTIONS % pformat(restrictions, width=60)
    return u''


def get_unique_restrictions(values, index):
    conditions = [obj['conditions'] for obj in values]

    if len(conditions) == 1:
        return filter_empty_fields(conditions[0])

    result = {}
    orig = conditions[int(index)]
    count = len(conditions)
    for key, value in orig.items():
        match = filter(lambda obj: key in obj and obj[key] == value, conditions)
        if len(match) == count:
            continue
        result[key] = value

    return filter_empty_fields(result)


def get_shared_restrictions(values):
    conditions = [obj['conditions'] for obj in values]

    if len(conditions) == 1:
        return {}

    result = {}
    count = len(conditions)
    for key, value in conditions[0].items():
        matched = len(filter(lambda obj: key in obj and obj[key] == value, conditions))
        if matched == count:
            result[key] = value

    return filter_empty_fields(result)


def _safe_getitem(dict_obj, path):
    if isinstance(dict_obj, dict):
        if '.' in path:
            name, path = path.split('.', 1)
            return _safe_getitem(dict_obj.get(name), path)
        else:
            return dict_obj.get(path, '')
    elif isinstance(dict_obj, (jinja2.Undefined, type(None))):
        return ''
    raise KeyError('Cannot found key "%s" in object "%s"' % (path, dict_obj))


def format_issue(project, format_description=True, **kwargs):
    params = dict(
        show_restrictions=show_restrictions,
        get_unique_restrictions=get_unique_restrictions,
        get_shared_restrictions=get_shared_restrictions,
        _=_safe_getitem,
        meta={'project': project},
    )
    params.update(kwargs)
    summary_template = app.jinja_env.from_string(get_summary_template())
    summary = summary_template.render(params)

    if format_description:
        description_template = app.jinja_env.from_string(get_description_template())
        description = description_template.render(params)
    else:
        description = None

    return summary, description


def format_comment(**kwargs):
    params = dict(
        show_restrictions=show_restrictions,
        get_unique_restrictions=get_unique_restrictions,
        get_shared_restrictions=get_shared_restrictions,
        _=_safe_getitem,
    )
    params.update(kwargs)
    comment_template = app.jinja_env.from_string(get_comment_template())
    comment = comment_template.render(params)

    return comment


def find_issues(summary):
    try:
        issues = client.issues.find(filter={'queue': STARTREK_QUEUE, 'summary': summary})
        issues = filter(lambda issue: issue.summary == summary, issues)
        return issues
    except exceptions.StartrekServerError:
        logger.exception('Cannot find ticket')
    return []


def create_issue(summary, description):
    if not STARTREK_ENABLED:
        return None

    issue = client.issues.create(
        queue=STARTREK_QUEUE,
        summary=summary,
        description=description,
        createdBy=get_ticket_author(),
        assignee=get_ticket_assignee(),
        followers=get_ticket_followers(),
        type=STARTREK_ISSUE_TYPE,
    )
    return issue


def create_issue_comment(issue, comment):
    if not STARTREK_ENABLED:
        return None

    text = (u"<{ настройки эксперимента в среде **%(yenv)s** отредактировал кто:%(author)s\n%(comment)s\n}>" %
            dict(yenv=yenv.type.capitalize(), author=request.yauser.login, comment=comment))
    comment = issue.comments.create(text=text)
    return comment


SUMMARY = u'[ {{meta.project}} ][ {{_id}} ]'

DESCRIPTION = u"""**Описание**

**Что ожидаем**

**Критерий выкатки**


#|
|| **Ссылка на тест-админку** 
 | ((https://localization-admin.tst.phone.yandex.net/localizations/edit/?url=/localizations/?project=launcher&project=launcher&id={{_id}} {{_id}})) ||
|| **Ссылка на прод-админку** 
 | ((https://localization-admin.phone.yandex.net/localizations/edit/?url=/localizations/?project=launcher&project=launcher&id={{_id}} {{_id}})) ||
|| **Соль** | {{ _(options, 'audience_salt') }} ||
{% for item in values %}
|| **Выборка {{ loop.index }}** 
 | ##{{ _(item, 'value') }}## 
 | {{ _(item, 'conditions.audience') }} 
 | {{ _(item, 'conditions.audience_offset') }} 
 | {{ show_restrictions(get_unique_restrictions(values, loop.index0)) }} ||
{% endfor %}
|| **Наблюдение** 
 | ((https://ab.yandex-team.ru/observation/<observation_id>/launch/geo_abt <observation_id> ))||
|| **Общие ограничения** 
 | {{ show_restrictions(get_shared_restrictions(values)) }} ||
|#
++* все время отображается в UTC++
&#x1F680;
"""

COMMENT = u""" 
#|
|| **Соль** 
| {{ _(options, 'audience_salt') }} ||
{% for item in values %}
|| **Выборка {{ loop.index }}** 
 | ##{{ _(item, 'value') }}## 
 | {{ _(item, 'conditions.audience') }} 
 | {{ _(item, 'conditions.audience_offset') }} 
 | {{ show_restrictions(get_unique_restrictions(values, loop.index0)) }} ||
{% endfor %}
|| **Общие ограничения** 
 | {{ show_restrictions(get_shared_restrictions(values)) }} ||
|#
++* все время отображается в UTC++
"""
