# encoding: utf-8
from flask import request
from werkzeug.exceptions import Forbidden
from localization_admin.models.permissions import Permission, Roles
from mongoengine.context_managers import switch_collection


def get_admins():
    with switch_collection(Permission, 'admins') as Admin:
        return [a.id for a in Admin.objects.all()]


def get_role(resource_name=None):
    if request.admin_mode:
        return Roles.Moderator
    if not hasattr(request.yauser, 'role'):
        universal_permission = Permission.objects.filter(pk=request.yauser.login).first()
        universal_role = universal_permission and universal_permission.role or Roles.Nobody
        resource_role = Roles.Nobody
        if resource_name:
            with switch_collection(Permission, resource_name + '._permissions') as ResourcePermission:
                resource_permission = ResourcePermission.objects.filter(pk=request.yauser.login).first()
            resource_role = resource_permission and resource_permission.role or Roles.Nobody
        setattr(request.yauser, 'role', Roles.most_privileged((universal_role, resource_role)))
    return getattr(request.yauser, 'role')


def request_view(view):
    if view.resource_name is None:
        return
    if (view.security_level not in ['public', 'protected']
       and get_role(view.resource_name) == Roles.Nobody):
        raise Forbidden(description='You are not allowed to browse %s' % (view.resource_name,))


def request_edit(view):
    request.read_only = False
    if view.resource_name is None:
        return
    if (view.security_level != 'public'
       and get_role(view.resource_name) not in (Roles.Moderator, Roles.Editor)):
        request.read_only = True
        if request.method != 'GET':
            raise Forbidden(description='You are not allowed to edit %s' % (view.resource_name,))


def can_edit_permissions(view, resource_name=None):
    resource_name = not resource_name and view.resource_name or resource_name
    if resource_name == 'admins':
        return request.admin_mode
    return get_role(resource_name) == Roles.Moderator
