from datetime import datetime, timedelta

from flask.ext.admin.contrib.mongoengine.fields import ModelFormField
from flask.ext.admin.model.fields import InlineFieldList
from werkzeug.datastructures import FileStorage
from wtforms.compat import string_types
from wtforms.fields import SelectFieldBase, Field, DateTimeField, TextAreaField, _unset_value, FormField, BooleanField

from .widgets import (CustomMongoFileInput, CustomInlineFormWidget, CustomInlineFieldListWidget,
                      GeoCodesSelectWidget, CustomCheckboxInput)


def empty_list_to_none_filter(data):
    if isinstance(data, (list, tuple)) and not data:
        return None
    return data


def replace_validator(field, old_validator_class, new_validator_class):
    """
        All instances of one validator class are replaced with instances of another.
    """
    validators = getattr(field, 'validators', [])

    for index, validator in zip(range(len(validators)), validators):
        if isinstance(validator, old_validator_class):
            field.validators[index] = new_validator_class()

    if getattr(field, '__iter__', None) and not isinstance(field, (SelectFieldBase,)):
        for subfield in field:
            replace_validator(subfield, old_validator_class, new_validator_class)


def fix_listfields_ids(field, prefixes=tuple()):
    """
        flask.ext.admin.contrib.mongoengine contains error, which does not
        allow nested listfields (Javascript on page creates new entries with wrong ids and
        obviously they are discarded)
        This function is small hack to fix this situation
    """
    prefixes = list(prefixes)

    if isinstance(field, InlineFieldList):
        for prefix in prefixes:
            if field.id.startswith(prefix):
                field.id = field.id[len(prefix) + 1:]  # assert len(separator) == 1

        for subfield in field:
            prefixes.append(subfield.id)
            fix_listfields_ids(subfield, prefixes)
            prefixes.pop()

    elif getattr(field, '__iter__', None) and not isinstance(field, (SelectFieldBase,)):
        for subfield in field:
            fix_listfields_ids(subfield, prefixes)


def trim_field(field, ordered_fields):
    if ordered_fields is True:
        return

    if isinstance(field, InlineFieldList):
        for subfield in field:
            trim_field(subfield, ordered_fields)

        trim_field(field.template, ordered_fields)

    elif isinstance(field, FormField):

        for subfield in field.form:

            if subfield.short_name in ordered_fields:
                trim_field(subfield, ordered_fields[subfield.short_name])
            elif subfield.data is None:
                field.form.__delitem__(subfield.short_name)
            else:
                try:
                    default = subfield.default()
                except TypeError:
                    default = subfield.default
                subfield.hidden = subfield.data == default


class FileStorageURLField(Field):
    """
        Field that handles file upload.
    """
    widget = CustomMongoFileInput()

    def __init__(self, file_storage, file_rename=lambda x: x, **kwargs):
        self.file_storage = file_storage
        self.file_rename = file_rename
        self.data_key = None
        super(FileStorageURLField, self).__init__(**kwargs)

    def process_formdata(self, valuelist):
        if not valuelist:
            self.data, self.data_key = None, None
            return
        for value in valuelist:
            if isinstance(value, FileStorage) and value:  # new file!
                self.data_key = self.file_storage.upload(self.file_rename(value.filename), value.stream.read())
                self.data = self.file_storage.get_link(self.data_key)
                return

            if isinstance(value, string_types):
                self.data = value
                try:
                    self.data_key = self.file_storage.get_key(value)
                except:
                    if self.data_key:
                        if self.file_storage.get_link(self.data_key) != self.data:
                            raise RuntimeError("Unable to determine filestorage key")

    def process_data(self, value):
        self.data_key = value
        if self.data_key:
            self.data = self.file_storage.get_link(self.data_key)
        else:
            self.data_key, self.data = None, None

    def populate_obj(self, obj, name):
        setattr(obj, name, self.data_key)


class CrxUploadField(FileStorageURLField):
    def process_formdata(self, valuelist):
        self._upload = None
        if not valuelist:
            self.data, self.data_key = None, None
            return
        for value in valuelist:
            if isinstance(value, FileStorage) and value:
                self.data, self.data_key = None, None
                self._upload = value
                return
        super(CrxUploadField, self).process_formdata(valuelist)


class CustomModelFormField(ModelFormField):
    """
        By default FormField does not support validators
        But logic requires some analogue of wtforms.validators.Optional
    """

    widget = CustomInlineFormWidget()

    def __init__(self, model, view, form_class, form_opts=None, required=False, **kwargs):
        self.required = required
        super(CustomModelFormField, self).__init__(model, view, form_class, form_opts, **kwargs)

    def validate(self, form, extra_validators=tuple()):

        form_validation_result = super(CustomModelFormField, self).validate(form, extra_validators)
        if self.required:
            return form_validation_result
        else:  # field is optional, if it is blank, we should ignore form validation
            if form_validation_result:
                return True
            else:
                if self.data is None:
                    return True
                else:
                    return False

    @property
    def data(self):
        return dict([(a[0], a[1]) for a in self.form.data.items() if a[1] is not None]) or None

    def populate_obj(self, obj, name):
        if self.data is not None:
            return super(CustomModelFormField, self).populate_obj(obj, name)
        else:
            setattr(obj, name, None)


class CustomInlineFieldList(InlineFieldList):
    widget = CustomInlineFieldListWidget()

    def __init__(self, *args, **kwargs):
        kwargs['default'] = list()
        super(CustomInlineFieldList, self).__init__(*args, **kwargs)

    @property
    def data(self):
        return [entry for entry in super(CustomInlineFieldList, self).data if entry is not None] or None

    def populate_obj(self, obj, name):
        _fake = type(str('_fake'), (object,), {})
        fake_obj = _fake()
        super(CustomInlineFieldList, self).populate_obj(fake_obj, name)
        setattr(obj, name, getattr(fake_obj, name) or None)

    def _add_entry(self, formdata, data=_unset_value, index=None):
        assert not self.max_entries or len(self.entries) < self.max_entries, \
            'You cannot have more than max_entries entries in this FieldList'
        new_index = self.last_index = index or (self.last_index + 1)
        name = '%s-%d' % (self.short_name, new_index)
        _id = '%s-%d' % (self.id, new_index)
        label = '%s-%d' % (self.short_name[:-1].capitalize() if self.short_name.endswith('s')
                           else self.short_name.capitalize(), new_index + 1)
        field = self.unbound_field.bind(form=None, name=name, label=label, prefix=self._prefix, id=_id)
        field.process(formdata, data)
        self.entries.append(field)
        return field


class SingleValueFieldList(CustomInlineFieldList):
    def __call__(self, *args, **kwargs):
        kwargs.update({'single_value': True})
        return super(CustomInlineFieldList, self).__call__(*args, **kwargs)


class SortableFieldList(CustomInlineFieldList):
    sortable = True


class TimezoneDateTimeField(DateTimeField):
    def __init__(self, time_shift=timedelta(hours=3), *args, **kwargs):
        self.time_shift = time_shift
        super(TimezoneDateTimeField, self).__init__(*args, **kwargs)

    def populate_obj(self, obj, name):
        if isinstance(self.data, datetime):
            setattr(obj, name, self.data - self.time_shift)
        else:
            super(TimezoneDateTimeField, self).populate_obj(obj, name)

    def process_data(self, data):
        if isinstance(data, datetime):
            data += self.time_shift
        super(TimezoneDateTimeField, self).process_data(data)


class CustomTextAreaField(TextAreaField):
    def process_formdata(self, valuelist):
        if valuelist:
            self.data = valuelist[0]
        else:
            self.data = None


class ClidsSetField(CustomInlineFieldList):
    def __init__(self, *args, **kwargs):
        kwargs.update({'default': tuple()})
        super(ClidsSetField, self).__init__(*args, **kwargs)

    def populate_obj(self, obj, name):
        candidate = [entry.data for entry in self.entries] or None
        setattr(obj, name, candidate)


class GeoCodesField(Field):
    widget = GeoCodesSelectWidget()

    def __init__(self, *args, **kwargs):
        super(GeoCodesField, self).__init__(*args, **kwargs)

    def process_data(self, data):
        self.data = data or None

    def process_formdata(self, formdata):
        if formdata and formdata[0]:
            self.data = [int(a) for a in formdata[0].split(',')]
        else:
            self.data = None


class CustomBooleanField(BooleanField):
    widget = CustomCheckboxInput()

    def process_formdata(self, valuelist):
        if valuelist:
            if valuelist[0] in self.false_values:
                self.data = False
            else:
                self.data = True
        else:
            try:
                self.data = bool(self.default())
            except TypeError:
                self.data = bool(self.default)
