# encoding: utf-8
from flask.ext.admin import form
from flask.ext.admin.contrib.mongoengine.form import CustomModelConverter as MongoModelConverter, get_form
from flask.ext.admin.form import BaseForm, FormOpts
from flask.ext.admin.form.widgets import Select2Widget
from flask.ext.admin.model.fields import InlineFieldList
from flask.ext.mongoengine.wtf import orm, fields as mongo_fields
from flask.ext.mongoengine.wtf.fields import ModelSelectField
from wtforms.fields import FormField, FieldList, TextAreaField, SelectField, SelectFieldBase, SelectMultipleField, \
    BooleanField
from wtforms.validators import Required as OldRequired, Optional, Optional as OldOptional

from .fields import replace_validator, fix_listfields_ids, trim_field, FileStorageURLField, ClidsSetField, \
    CustomModelFormField, CustomInlineFieldList, TimezoneDateTimeField, SingleValueFieldList, CustomTextAreaField, \
    empty_list_to_none_filter, GeoCodesField, SortableFieldList, CustomBooleanField
from .helpers import coerce_if_not_none
from .validators import Required as NewRequired, PrimaryKey, StrictOptional as NewOptional, URL, HttpsURL, ValidObjectId
from .widgets import WidgetConverter, TaggedStringWidget


class CustomBaseForm(BaseForm):
    """
        Custom base class for form scaffolding
    """

    def __init__(self, formdata=None, obj=None, prefix='', ordered_fields=None, **kwargs):
        if formdata:
            obj = None  # preventing model from preserving values for fields hidden by js
        super(CustomBaseForm, self).__init__(formdata=formdata, obj=obj, prefix=prefix, **kwargs)
        for field in self:
            """
                Replacing validators: library Required does not allow 0 and False as value of required field.
            """
            replace_validator(field, OldRequired, NewRequired)
            replace_validator(field, OldOptional, NewOptional)
            """
                Library bug : nested listfields are unable to add entries. Small hack, fixing it.
            """
            fix_listfields_ids(field)
            """
                Dynamically hide fields from user.
            """
            if ordered_fields is not None:
                if field.short_name in ordered_fields:  # show field at least partly
                    trim_field(field, ordered_fields[field.short_name])
                elif field.data is None:  # field is both hidden and blank. Delete it from form.
                    self.__delitem__(field.short_name)
                else:  # field is hidden, but has some value (default or set by superuser). If latter - show it
                    try:
                        default = field.default()
                    except TypeError:
                        default = field.default

                    field.hidden = field.data == default  # Nothing to see here for user


class CustomModelConverter(MongoModelConverter):
    def convert(self, model, field, field_args):
        conversion_result = super(CustomModelConverter, self).convert(model, field, field_args)
        if getattr(field, 'disabled', False):
            if 'widget_conversion_args' not in conversion_result.kwargs:
                conversion_result.kwargs['widget_conversion_args'] = dict()
            conversion_result.kwargs['widget_conversion_args']['disabled'] = 'disabled'
        if getattr(field, 'readonly', False):
            if 'widget_conversion_args' not in conversion_result.kwargs:
                conversion_result.kwargs['widget_conversion_args'] = dict()
            conversion_result.kwargs['widget_conversion_args']['readonly'] = 'readonly'
        if conversion_result.field_class == InlineFieldList:
            conversion_result.field_class = CustomInlineFieldList
        if conversion_result.field_class == TextAreaField:
            conversion_result.field_class = CustomTextAreaField
        if conversion_result.field_class == BooleanField:
            conversion_result.field_class = CustomBooleanField
        if conversion_result.field_class == SelectField:
            conversion_result.kwargs['coerce'] = coerce_if_not_none(conversion_result.kwargs['coerce'])
        if issubclass(conversion_result.field_class, (SelectFieldBase,)):
            if conversion_result.field_class in (SelectMultipleField,
                                                 mongo_fields.ModelSelectMultipleField):
                multiple = True
                conversion_result.kwargs['filters'] = [empty_list_to_none_filter]
            else:
                multiple = False
            conversion_result.kwargs['widget'] = Select2Widget(multiple=multiple)
        if hasattr(field, 'primary_key') and field.primary_key:
            conversion_result.kwargs['validators'].append(PrimaryKey(model))

        # wtforms.Optional validator skips validation if field.raw_data is None
        # Sadly, it's ALWAYS the case for complex fields like FieldList and FormField
        if issubclass(conversion_result.field_class, (FieldList, FormField)):
            conversion_result.kwargs.update({'validators': [validator for validator in
                                                            conversion_result.kwargs.get('validators', [])
                                                            if not isinstance(validator, Optional)]})

        conversion_result.kwargs['widget'] = WidgetConverter(
            conversion_result.kwargs.get('widget') or conversion_result.field_class.widget,
            field, conversion_result.kwargs.pop('widget_conversion_args', None))

        return conversion_result

    @orm.converts('PropertyReferenceField')
    def conv_PropertyReference(self, model, field, kwargs):
        return ModelSelectField(model=field.document_type,
                                queryset=field.document_type.objects.filter(
                                    **{field.field_name + '__exists': True}).no_cache(), **kwargs)

    @orm.converts('ObjectIdField')
    def conv_ObjectId(self, model, field, kwargs):
        validators = kwargs.get('validators', [])
        validators.append(ValidObjectId())
        kwargs['validators'] = validators
        return TextAreaField(**kwargs)

    @orm.converts('TaggedStringField')
    def conv_TaggedString(self, model, field, kwargs):
        result = self.conv_String(model, field, kwargs)
        result.kwargs['widget'] = TaggedStringWidget(field.tags)
        return result

    @orm.converts('TimestampDateTimeField')
    def conv_TimestampDateTime(self, model, field, kwargs):
        kwargs['widget'] = form.DateTimePickerWidget()
        return TimezoneDateTimeField(**kwargs)

    @orm.converts('DateTimeField')
    def conv_TimezoneDateTime(self, model, field, kwargs):
        kwargs['widget'] = form.DateTimePickerWidget()
        return TimezoneDateTimeField(**kwargs)

    @orm.converts('SingleValueListField')
    def conv_SingleValueList(self, model, field, kwargs):
        result = self.conv_List(model, field, kwargs)
        result.field_class = SingleValueFieldList
        return result

    @orm.converts('SortableListField')
    def conv_SortableList(self, model, field, kwargs):
        result = self.conv_List(model, field, kwargs)
        result.field_class = SortableFieldList
        return result

    @orm.converts('SerializedListField')
    def conv_SerializedList(self, model, field, kwargs):
        result = self.conv_List(model, field, kwargs)
        result.field_class = CustomInlineFieldList
        return result

    @orm.converts('GeoCodesListField')
    def conv_GeoCodesList(self, model, field, kwargs):
        default_value = kwargs.get('default', None)
        if callable(default_value):
            default_value = default_value()
        if not default_value:  # overriding empty list with None, but sufficient default values are allowed
            kwargs['default'] = lambda: None
        return GeoCodesField(**kwargs)

    @orm.converts('ListField')
    def conv_List(self, model, field, kwargs):
        if field.field is None:
            raise ValueError('ListField "%s" must have field specified for model %s' % (field.name, model))

        # Disabling multi select because of problems with sorting references. Might become usefull later
        """
        if isinstance(field.field, ReferenceField):
            loader = getattr(self.view, '_form_ajax_refs', {}).get(field.name)
            if loader:
                return AjaxSelectMultipleField(loader, **kwargs)

            kwargs['widget'] = form.Select2Widget(multiple=True)

            # TODO: Support AJAX multi-select (This is TODO from library)
            doc_type = field.field.document_type
            return mongo_fields.ModelSelectMultipleField(model=doc_type, **kwargs)
        """

        # Create converter
        view = self._get_subdocument_config(field.name)
        converter = self.clone_converter(view)

        if field.field.choices:
            kwargs['multiple'] = True
            return converter.convert(model, field.field, kwargs)

        unbound_field = converter.convert(model, field.field, {})
        min_entries = 0
        if field.required:
            min_entries = 1
        return InlineFieldList(unbound_field, min_entries=min_entries, **kwargs)

    @orm.converts('MongoFileStorageURLField')
    def conv_FileStorageURL(self, model, field, kwargs):
        if 'validators' not in kwargs:
            kwargs['validators'] = list()
        kwargs['validators'].append(HttpsURL())
        return FileStorageURLField(field.file_storage, field.file_rename, **kwargs)

    @orm.converts('URLField')
    def conv_URL(self, model, field, kwargs):
        kwargs['validators'].append(URL())
        self._string_common(model, field, kwargs)
        return mongo_fields.NoneStringField(**kwargs)

    @orm.converts('MongoClidsSetField')
    def conv_ClidsSet(self, model, field, kwargs):
        view = self._get_subdocument_config(field.name)
        converter = self.clone_converter(view)
        unbound_field = converter.convert(model, field.field, {})
        min_entries = 0
        if field.required:
            min_entries = 1
        return ClidsSetField(unbound_field, min_entries=min_entries, **kwargs)

    @orm.converts('EmbeddedDocumentField')
    def conv_EmbeddedDocument(self, model, field, kwargs):

        required = False
        if [validator for validator in kwargs.get('validators', []) if isinstance(validator, OldRequired)]:
            required = True

        kwargs['validators'] = []

        view = self._get_subdocument_config(field.name)

        form_opts = FormOpts(widget_args=getattr(view, 'form_widget_args', None),
                             form_rules=view._form_rules)

        form_class = view.get_form()
        if form_class is None:
            converter = self.clone_converter(view)
            form_class = get_form(field.document_type_obj, converter,
                                  base_class=view.form_base_class or BaseForm,
                                  only=view.form_excluded_columns,
                                  field_args=view.form_args,
                                  extra_fields=view.form_extra_fields)

            form_class = view.postprocess_form(form_class)

        return CustomModelFormField(field.document_type_obj, view, form_class, form_opts=form_opts,
                                    required=required, **kwargs)
