# coding: utf-8
from datetime import datetime

import pytz
from flask import request, url_for, flash
from wtforms.fields import HiddenField

from localization_admin import log as logging
from localization_admin.models.localization import LocalizationData
from localization_admin.models.helpers import get_localization_projects
from localization_admin.views.base_model import BaseCrossCollectionModelView
from localization_admin.views.helpers import get_localization_ordered_fields
from localization_admin.models.support_info.startrek_interaction import StartrekInteraction
from localization_admin.models.helpers import to_dict, format_diff

from localization_admin.startrek import (
    format_issue, create_issue, find_issues, create_issue_comment, prepare_params, format_comment
)

logger = logging.getLogger(__name__)

tz_moscow = pytz.timezone('Europe/Moscow')


def are_models_equals(model1, model2):
    dict1 = to_dict(model1)
    dict2 = to_dict(model2)
    if dict1 and 'updated_at' in dict1:
        del dict1['updated_at']
    if dict2 and 'updated_at' in dict2:
        del dict2['updated_at']
    return cmp(dict1, dict2) == 0


class ProjectLocalizationView(BaseCrossCollectionModelView):
    """
        Common view class for all projects
    """
    column_list = ['id', 'updated_at']
    column_searchable_list = ['id']
    column_type_formatters = {
        datetime: lambda view, value: tz_moscow.fromutc(value).strftime('%Y-%d-%m %H:%M:%S'),
        type(None): lambda view, value: "-",
    }

    form_overrides = {'updated_at': HiddenField}

    list_template = 'localizations/list.html'
    create_template = 'localizations/create.html'
    edit_template = 'localizations/edit.html'

    security_level = 'protected'

    @property
    def create_form_options(self):
        return {'ordered_fields': get_localization_ordered_fields(self.resource_name)}

    @property
    def edit_form_options(self):
        return {'ordered_fields': get_localization_ordered_fields(self.resource_name)}

    def _handle_view(self, name, **kwargs):
        super(ProjectLocalizationView, self)._handle_view(name, **kwargs)
        if 'project' not in request.args:
            projects = get_localization_projects()
            list_info = [
                dict(
                    url='/localizations/?project=%s' % project,
                    title='Project %s localization data' % project
                )
                for project in projects
            ]
            list_info.sort(key=lambda x: x['title'])
            return self.render('link_list.html', list_info=list_info)

    def on_model_imported(self, model_before, model_after):
        try:
            self.log_model_import_to_startrek(model_before, model_after)
        except:
            logger.exception('cannot log model import to Startrek')

    def log_model_import_to_startrek(self, model_before, model_after):
        # handle here only POST method
        if not model_after:
            return

        project = request.args.get('project')
        integration_applicable = \
            StartrekInteraction.is_applicable(project=project, key_name=model_after.id)

        if not integration_applicable:
            return

        template_params = prepare_params(to_dict(model_after))
        try:
            summary, description = format_issue(project=project, **template_params)
        except:
            logger.exception('cannot format issue')
            flash('Cannot format summary/description for Startrek ticket', 'error')
            return

        issues = find_issues(summary)
        if len(issues) > 1:
            flash('More than one ticket found in Startrek', 'warning')
            return

        issue = issues[0] if issues else None
        if are_models_equals(model_before, model_after):
            flash('The document is not changed because it`s identical', 'info')
            return

        flash('document successfully imported!', 'info')

        if issue:
            # experiment and issue are exists before
            comment_text = u'Эксперимент успешно ипортирован!\n\n' + format_comment(**template_params)
            create_issue_comment(issue, comment=comment_text)
            flash('Into Startrek ticket has been added comment about imported document', 'info')
        else:
            description = u'Эксперимент успешно ипортирован!\n\n' + description
            create_issue(summary, description)
            flash('Startrek ticket has been created for experiment', 'info')

    def on_model_delete(self, model):
        pass

    def on_model_change(self, form, model, is_created):
        super(ProjectLocalizationView, self).on_model_change(form, model, is_created)
        try:
            self.log_model_change_to_startrek(model)
        except:
            logger.exception('cannot log action to Startrek')

    def log_model_change_to_startrek(self, model):
        project = request.args.get('project')
        integration_applicable = \
            StartrekInteraction.is_applicable(project=project, key_name=model.id)

        if integration_applicable:
            template_params = prepare_params(to_dict(model))
            try:
                summary, description = format_issue(project=project, **template_params)
            except:
                logger.exception('cannot format issue')
                flash('Cannot format summary/description for Startrek ticket', 'error')
                return

            issues = find_issues(summary)
            if len(issues) > 1:
                flash('More than one ticket found in Startrek', 'warning')
                return

            issue = issues[0] if issues else None
            found_model = self.model.objects(pk=model.pk).first()
            if are_models_equals(model, found_model):
                return

            if found_model and issue:
                # experiment and issue are exists
                document_before = found_model.to_json() if found_model else ''
                document_after = model.to_json()
                comment_text = format_comment(**template_params)
                comment_text += (u'\n\n<{ посмотреть изменения:\n##<# %s #>##\n}>\n' %
                                 format_diff(document_before, document_after))
                create_issue_comment(issue, comment=comment_text)

                flash('Into Startrek ticket has been added comment about changes', 'info')
            else:
                # experiment does not exists
                if issue:
                    flash('Startrek ticket already exists', 'info')
                else:
                    create_issue(summary, description)
                    flash('Startrek ticket has been created for experiment', 'info')

    @property
    def _template_args(self):
        request_args = request.args
        args = super(ProjectLocalizationView, self)._template_args
        args.update({'project': request_args.get('project', None)})
        args['title'] = 'Localization data for %s' % self.resource_name
        args['allow_startrek_action'] = StartrekInteraction.is_applicable
        if 'project' in request_args and 'id' in request_args:
            integration_applicable = StartrekInteraction.is_applicable(project=request_args.get('project'),
                                                                       key_name=request_args.get('id'))
            if '.export_view' not in request.endpoint and integration_applicable:
                flash('All actions with this project/id will be logged into Startrek!', 'info')

        return args

    @property
    def resource_name(self):
        return request and request.args.get('project', None) or None

    @property
    def logging_prefix(self):
        return request and request.args.get('project', 'localizations').capitalize() or 1

    @property
    def _logging_prefix(self):
        return request.args.get('project', 'localizations').capitalize()

    def get_collection_name(self, request):
        project = request.args.get('project')
        return 'localization.' + project

    def _get_url(self, view=None, page=None, sort=None, sort_desc=None,
                 search=None, filters=None, project=None):
        """
            Generate page URL with current page, sort column and
            other parameters.

            :param view:
                View name
            :param page:
                Page number
            :param sort:
                Sort column index
            :param sort_desc:
                Use descending sorting order
            :param search:
                Search query
            :param filters:
                List of active filters
        """
        if not search:
            search = None

        if not page:
            page = None

        if not project:
            project = request.args.get('project')

        kwargs = dict(page=page, sort=sort, desc=sort_desc, search=search, project=project)

        if filters:
            for i, pair in enumerate(filters):
                idx, value = pair

                key = 'flt%d_%s' % (i, self.get_filter_arg(idx, self._filters[idx]))
                kwargs[key] = value

        return url_for(view, **kwargs)


view = ProjectLocalizationView(LocalizationData, 'Project localization', endpoint='localizations')
