from flask_mongoengine import MongoEngine, BaseQuerySet
from mongoengine import signals

from yaphone.localization_admin.src.helpers import BaseDecoratorMeta, autoreconnect
import yaphone.localization_admin.src.models.fields as fields
import yaphone.localization_admin.src.log as logging

log = logging.getLogger('deltas')

db = MongoEngine()


def log_changes_on_save(sender, document, created):
    if created:
        log.info(
            "Document created. Collection: %s, JSON: %s",
            document._get_collection_name(),
            document.to_json(),
        )
    else:
        log.info(
            "Document updated. Collection: %s, Old: %s, New: %s",
            document._get_collection_name(),
            sender.objects.get(id=document.id).to_json(),
            document.to_json(),
        )


def log_changes_on_delete(sender, document):
    log.info(
        "Document deleted. Collection: %s, JSON: %s",
        document._get_collection_name(),
        document.to_json(),
    )


signals.pre_save_post_validation.connect(log_changes_on_save)
signals.post_delete.connect(log_changes_on_delete)


class AutoReconnectMeta(BaseDecoratorMeta):

    @staticmethod
    def decorate(f):
        return autoreconnect(f)


class QuerySet(BaseQuerySet):
    __metaclass__ = AutoReconnectMeta


class Document(db.Document):
    meta = {'abstract': True,
            'queryset_class': QuerySet}


class DynamicDocument(db.DynamicDocument):
    meta = {'abstract': True,
            'queryset_class': QuerySet}


for field_class_name in fields.__all__:
    setattr(db, field_class_name, getattr(fields, field_class_name))


db.DynamicDocument = DynamicDocument
db.Document = Document
