import datetime
from calendar import timegm

import mongoengine as mongo
from mongoengine.base.fields import BaseField


CLID_REGEX = "clid\d+"

__all__ = [
    'MongoFileStorageURLField', 'MongoClidsSetField', 'SingleValueListField', 'SortableListField',
    'GeoCodesListField', 'TaggedStringField', 'ServiceLayoutReferenceField', 'PropertyReferenceField',
    'URLField', 'TimestampDateTimeField',
]


class Clid(mongo.EmbeddedDocument):
    name = mongo.StringField(required=True, regex=CLID_REGEX, help_text="Clid's ID")
    value = mongo.StringField(required=True, help_text="Clid's value")


class MongoFileStorageURLField(mongo.StringField):
    """
        Can store "raw" links as any StringField,
        additionally can upload files to file storage and store links on them.
    """
    def __init__(self, file_storage, file_rename=lambda x: x, **kwargs):
        self.file_storage = file_storage
        self.file_rename = file_rename
        super(MongoFileStorageURLField, self).__init__(**kwargs)


class MongoClidsSetField(mongo.StringField):
    """
        Class is used for conversion (back and forth) from request formdata like:
        [clid_pair1, ..., clid_pairN], where clid_pair = {'name': 'clid\d+', value: '\d+'}
        to stored in db string like:
        "clid_pair1.name:clid_pair1.value,....,clid_pairN.name:clid_pairN.value"
    """

    def __init__(self, regex=None, max_length=None, min_length=None, **kwargs):
        self.field = mongo.EmbeddedDocumentField(Clid)
        super(MongoClidsSetField, self).__init__(regex, max_length, min_length, **kwargs)

    def __get__(self, instance, owner):
        value = super(MongoClidsSetField, self).__get__(instance, owner)
        result = []
        if value:
            for clid_pair in value.split(','):
                if clid_pair.count(':') == 1:
                    clid_name, clid_value = clid_pair.split(':')
                    result.append({'name': clid_name, 'value': clid_value})
        return result

    def __set__(self, instance, value):
        new_value = value
        if isinstance(value, (tuple, list)):
            value = [clid_pair for clid_pair in value if clid_pair]
            new_value = None
            clids = sorted(value, key=lambda clid_pair: clid_pair['name'])
            if clids:
                new_value = ','.join([clid_pair['name'] + ':' + clid_pair['value'] for clid_pair in clids])
        return super(MongoClidsSetField, self).__set__(instance, new_value)


class SingleValueListField(mongo.ListField):
    pass  # TODO something


class SortableListField(mongo.ListField):
    """
        ATTENTION
        before using make sure that js is working fine with embedded field!
    """


class GeoCodesListField(mongo.ListField):
    """
        field with ajax widget to show region names instead of just numbers
    """
    def __init__(self, *args, **kwargs):
        super(GeoCodesListField, self).__init__(mongo.IntField(), *args, **kwargs)


class TaggedStringField(mongo.StringField):
    """
        String with tags (select2 style)
    """
    single_value = True

    def __init__(self, *args, **kwargs):
        self.tags = kwargs.pop('tags')
        super(TaggedStringField, self).__init__(*args, **kwargs)


class ServiceLayoutReferenceField(mongo.ReferenceField):
    pass


class PropertyReferenceField(BaseField):
    """
        Reference to unique, but non-id field of other model
    """
    def __init__(self, document_type, field_name, **kwargs):
        if not issubclass(document_type, (mongo.Document,)):
            self.error("Argument to PropertyReferenceField constructor must be a "
                            "document class")

        if not document_type._fields[field_name].unique:
            self.error("Only unique properties may be referenced")
        self.document_type = document_type
        self.field_name = field_name
        super(PropertyReferenceField, self).__init__(**kwargs)

    def to_mongo(self, document):
        field = self.document_type._fields[self.field_name]
        if isinstance(document, mongo.Document):
            value = document._data.get(self.field_name)
        else:
            value = document
        return field.to_mongo(value)

    def to_python(self, value):
        if value is not None and not isinstance(value, (mongo.Document,)):
            try:
                value = self.document_type.objects.get(**{self.field_name: value})
            except:
                value = None
        return value

    def prepare_query_value(self, op, value):
        return (value is None and [None] or [self.to_mongo(value)])[0]

    def validate(self, value):
        if not isinstance(value, self.document_type):
            self.error("A PropertyReferenceField only accepts documents")
        if value.id is None:
            self.error("You can only reference properties of documents already saved to db")

    def lookup_member(self, member_name):
        return self.document_type._fields.get(member_name)


class URLField(mongo.URLField):
    """
        Overriding mongoengine URLField to prevent failing on model.save()
        values with exotic shemas and to stop passing CUSTOM_URL_REGEX to all models
    """
    def validate(self, value):
        pass  # all validation is expected on form - level


class TimestampDateTimeField(mongo.DateTimeField):
    def to_mongo(self, value):
        value = super(TimestampDateTimeField, self).to_mongo(value)
        if isinstance(value, datetime.datetime):
            return str(timegm(value.timetuple()))

    def to_python(self, value):
        if value:
            return datetime.datetime.utcfromtimestamp(int(value))

    def validate(self, value):
        pass


class SerializedListField(mongo.ListField):
    def to_python(self, value):
        if isinstance(value, str):
            return [item for item in value.split('|')] or None
        else:
            return value or None

    def to_mongo(self, value):
        if isinstance(value, list):
            return u'|'.join(value) or None
        else:
            return value or None
