import json
from datetime import datetime

from bson.objectid import ObjectId
from flask import flash, request, redirect, url_for
from flask_admin.actions import action
from flask_admin.babel import lazy_gettext
from flask_admin.base import expose
from flask_admin.contrib.mongoengine import ModelView
from flask_admin.form import FormOpts
from flask_admin.helpers import get_form_data, get_redirect_target
from flask_admin.model.helpers import get_mdict_item_or_list
from jinja2 import Markup
from mongoengine import ObjectIdField
from mongoengine.errors import NotUniqueError, ValidationError
from werkzeug.exceptions import Forbidden

import yenv

from yaphone.localization_admin.src import log as logging
from yaphone.localization_admin.src.models.helpers import to_dict
from yaphone.localization_admin.src.models.support_info.startrek_interaction import StartrekInteraction
from yaphone.localization_admin.src.startrek import (
    format_issue, find_issues, prepare_params
)
from yaphone.localization_admin.src.views import access, helpers as h2
from yaphone.localization_admin.src.views.form import CustomModelConverter, CustomBaseForm
from yaphone.localization_admin.src.views.widgets import WidgetDisabler

log = logging.getLogger('changes')
logger = logging.getLogger(__name__)


class CustomEncoder(json.JSONEncoder):
    def default(self, obj):
        if isinstance(obj, datetime):
            return {
                "_type": "datetime",
                "value": obj.isoformat(sep=' ')
            }
        return super(CustomEncoder, self).default(obj)


class CustomDecoder(json.JSONDecoder):
    ISO_FORMAT = '%Y-%m-%d %H:%M:%S'
    ISO_FORMAT_MS = ISO_FORMAT + '.%f'

    def __init__(self, *args, **kwargs):
        json.JSONDecoder.__init__(self, object_hook=self.object_hook, *args, **kwargs)

    def object_hook(self, obj):
        if '_type' not in obj:
            return obj
        if obj['_type'] == 'datetime':
            value = obj['value']
            return datetime.strptime(value, self.ISO_FORMAT_MS if '.' in value else self.ISO_FORMAT)
        return obj


class BaseModelView(ModelView):
    # Logging

    logging_prefix = None

    # Templates

    edit_template = 'base_model/edit.html'

    export_template = 'base_model/export.html'

    import_template = 'base_model/import.html'

    create_template = 'base_model/create.html'

    list_template = 'base_model/list.html'

    _create_form_options = {}

    _edit_form_options = {}

    @property
    def create_form_options(self):
        return self._create_form_options

    @property
    def edit_form_options(self):
        return self._edit_form_options

    @property
    def _template_args(self):
        # leting template know if user has permission to edit
        args = super(BaseModelView, self)._template_args
        args.update({'read_only': request.read_only, 'h2': h2,
                     'body_background_color': self.admin.app.config['BACKGROUND_COLOR']})
        return args

    # Form

    model_form_converter = CustomModelConverter

    form_base_class = CustomBaseForm

    # Id policy

    show_id_on_edit = False

    # Init
    def __init__(self, model, *args, **kwargs):
        if not self.logging_prefix:
            self.logging_prefix = self.__class__.__name__
        super(BaseModelView, self).__init__(model, *args, **kwargs)

    # Id check

    def check_id(self, _id):
        return self.model.objects.filter(pk=_id).first() is None

    # pregenerated value for id on create

    def create_form(self, obj=None, **kwargs):
        kwargs.update(self.create_form_options)
        form = self._create_form_class(get_form_data(), obj=obj, **kwargs)
        if form.id.data is None:
            form.id.data = h2.generate_id(self.check_id)
        return form

    # prevents id field from being displayed on edit

    def edit_form(self, obj=None, **kwargs):
        kwargs.update(self.edit_form_options)
        form = self._edit_form_class(get_form_data(), obj=obj, **kwargs)

        if not self.show_id_on_edit:
            del form.id
        else:
            form.id.data = getattr(obj, 'id', form.id.data)
            form.id.widget = WidgetDisabler(form.id.widget)
            form.id.validators = tuple()  # to prevent "Object with this id already exist" error

        return form

    def update_model(self, form, model):
        # even if id is displayed, you can't edit it
        if self.show_id_on_edit:
            del form.id

        try:
            form.populate_obj(model)
            model.validate()  # validate before call on_model_change
            self._on_model_change(form, model, False)
            model.save()
        except h2.db_errors as e:
            flash('Failed to update model: database is unavailable, try refreshing the page', 'error')
            logging.error("{prefix} | Failed to update {id}: {error}".format(prefix=self.logging_prefix,
                                                                             error=repr(e), id=model.id))
        except ValidationError as e:
            flash(str(e), 'error')
            logging.error("{prefix} | Validation failed {id}: {error}".format(prefix=self.logging_prefix,
                                                                              error=repr(e), id=model.id))
        except NotUniqueError as e:
            if not h2.set_error(self.model, form, e):
                flash('Failed to update model: one of the values is duplicate, while expected to be unique', 'error')
        except Exception as e:
            logging.error("{prefix} | Failed to update {id}: {error}".format(
                prefix=self.logging_prefix, id=model.id, error=repr(e)))
            raise
        else:
            self.after_model_change(form, model, False)
            log.info("%s | %s updated by %s" % (self.logging_prefix, model.id, request.yauser.login))
            return True
        return False

    def create_model(self, form, **kwargs):
        try:
            model = self.model()
            form.populate_obj(model)
            model.validate()  # validate before call on_model_change
            self._on_model_change(form, model, True, **kwargs)
            model.save(force_insert=True)
        except h2.db_errors as e:
            flash('Failed to create model: database in unavailable, try refreshing the page', 'error')
            logging.error("{prefix}| Failed to create: {error}".format(prefix=self.logging_prefix,
                                                                       error=repr(e)))
        except ValidationError as e:
            flash(str(e), 'error')
            logging.error("{prefix} | Validation failed {id}: {error}".format(prefix=self.logging_prefix,
                                                                              error=repr(e), id=model.id))
        except NotUniqueError as e:
            if not h2.set_error(self.model, form, e):
                flash('Failed to create model: one of the values is duplicate, while expected to be unique', 'error')
        except Exception as e:
            logging.error("{prefix}| Failed to create: {error}".format(prefix=self.logging_prefix,
                                                                       error=repr(e)))
            raise
        else:
            self.after_model_change(form, model, True)
            log.info("%s| %s created by %s" % (self.logging_prefix, model.id, request.yauser.login))
            return True
        return False

    def delete_model(self, model):
        result = super(BaseModelView, self).delete_model(model)
        if result:
            log.info("%s| %s deleted by %s" % (self.logging_prefix, model.id, request.yauser.login))
        else:
            logging.error("%s| Model delete failed for id %s by %s" %
                          (self.logging_prefix, model.id, request.yauser.login))
        return result

    # Mass model delete was failing on some id checks

    action_disallowed_list = ['delete']

    @action('delete',
            lazy_gettext('Delete'),
            lazy_gettext('Are you sure you want to delete selected models?'))
    def action_delete(self, ids):  # Do we need it? TODO
        raise NotImplementedError(description='Multiple removal is not implemented')

    @staticmethod
    def calc_hash(document):
        return hex(hash(json.dumps(document, cls=CustomEncoder, sort_keys=True)))

    @staticmethod
    def check_importing_document(doc):
        fields = ['_hash', 'document', '_project', '_exported_at', '_environment']
        for field in fields:
            if field not in doc:
                raise Exception('expected %s fields in imported document' % fields)

        hash = doc['_hash']
        doc['_hash'] = ''
        if BaseModelView.calc_hash(doc) != hash:
            raise Exception('Document has been changed (hash mismatch). Please, import original exported document '
                            'and than made appropriate changes')

    @expose('/export/', methods=('GET',))
    def export_view(self):
        pk = request.values.get('id')
        model = self.model.objects(pk=pk).first()
        if not model:
            flash(u'Cannot find document for export by its pk=%s' % str(pk), 'error')
            return_url = get_redirect_target() or url_for('.index_view', project=request.values.get('project'))
            return redirect(return_url)

        orig_document = to_dict(model)
        orig_document['id'] = orig_document['_id']
        del orig_document['_id']
        document = {
            '_hash': '',
            '_project': request.values.get('project'),
            '_exported_at': datetime.utcnow(),
            '_environment': yenv.type,
            'document': orig_document,
        }
        document['_hash'] = BaseModelView.calc_hash(document)

        return_url = get_redirect_target() or url_for('.index_view', project=request.values.get('project'))
        return self.render(self.export_template, document=json.dumps(document, cls=CustomEncoder, indent=4),
                           return_url=return_url)

    def get_importing_document(self):
        posted_document = request.values.get('document')
        if not posted_document:
            raise ('Document must be supplied!', 'warn')

        doc = json.loads(posted_document, cls=CustomDecoder)
        self.check_importing_document(doc)
        return self.model(**doc['document'])

    @expose('/import/', methods=('GET', 'POST'))
    def import_view(self):
        if not self.can_import:
            return redirect(url_for('.index_view', project=request.values.get('project')))

        return_url = get_redirect_target() or url_for('.index_view', project=request.values.get('project'))
        if request.method == 'GET':
            return self.render(self.import_template, return_url=return_url)

        try:
            posted_model = self.get_importing_document()
            before_model = self.model.objects(pk=posted_model.pk).first()
        except Exception as e:
            flash(Markup(u'failed import document!<br>%s' % str(e)), 'error')
            return redirect(url_for('.import_view', project=request.values.get('project')))

        try:
            posted_model.save()
        except Exception as e:
            log.exception('Cannot save imported document')
            flash(Markup(u'failed import document!<br>%s' % str(e)), 'error')
            return redirect(return_url)

        self.on_model_imported(before_model, posted_model)
        return redirect(url_for('.edit_view', project=request.values.get('project'), id=posted_model.pk))

    @expose('/ticket/', methods=('GET',))
    def ticket(self):

        return_url = get_redirect_target() or request.referrer

        # check parameters
        project = request.args.get('project')
        key = request.args.get('key')
        if not project or not key:
            return redirect(return_url)

        # is it applicable to StarTrek?
        integration_applicable = StartrekInteraction.is_applicable(project=project, key_name=key)
        if not integration_applicable:
            return redirect(return_url)

        model = self.get_one(key)
        template_params = prepare_params(to_dict(model))
        try:
            summary, _ = format_issue(project=project, format_description=False, **template_params)
        except:
            logger.exception('cannot format issue')
            flash('Cannot format summary/description for StarTrek ticket', 'error')
            return redirect(return_url)

        issues = find_issues(summary)
        if len(issues) > 1:
            flash('More than one ticket found in StarTrek', 'warning')
            return redirect(return_url)

        if not issues:
            flash('Cannot find appropriate StarTrek ticket by summary "%s"' % summary, 'warning')
            return redirect(return_url)

        url = 'https://st.yandex-team.ru/{ticket_id}'.format(ticket_id=issues[0].key)
        return redirect(url)

    def on_model_imported(self, before_model, model):
        flash('document successfully imported!', 'info')

    @expose('/copy/', methods=('GET', 'POST'))
    def copy_view(self):
        return_url = get_redirect_target() or url_for('.index_view')
        request_id = get_mdict_item_or_list(request.args, 'id')
        if request_id is None:
            return redirect(return_url)
        if not self.can_create:
            return redirect(return_url)
        model = self.get_one(request_id)
        if model is None:
            return redirect(return_url)
        if request.method == 'GET':
            form = self.create_form(obj=model)
            form_opts = FormOpts(widget_args=self.form_widget_args,
                                 form_rules=self._form_create_rules)
            if h2.is_ObjectId(form.id.data):
                form.id.data = h2.generate_id(self.check_id)
            else:
                num = 1
                while not self.check_id("%s-copy%s" % (form.id.data, num)):
                    num += 1
                form.id.data = "%s-copy%s" % (form.id.data, num)
            return self.render(self.create_template, form=form, form_opts=form_opts, return_url=return_url)
        # POST
        return self.create_view()

    # making search case insensitive
    def get_list(self, page, sort_column, sort_desc, search, filters,
                 execute=True, page_size=None):
        """
            Get list of objects from MongoEngine

            :param page:
                Page number
            :param sort_column:
                Sort column
            :param sort_desc:
                Sort descending
            :param search:
                Search criteria
            :param filters:
                List of applied filters
            :param execute:
                Run query immediately or not
        """
        if not sort_column:
            sort_column, sort_desc = 'pk', True
        if not search or search.startswith('*'):
            return super(BaseModelView, self).get_list(page, sort_column, sort_desc, search, filters,
                                                       execute)
        else:
            return super(BaseModelView, self).get_list(page, sort_column, sort_desc,
                                                       '*' + search, filters, execute)

    # support for ObjectId
    def get_one(self, _id):
        if isinstance(self.model.id, ObjectIdField):
            _id = ObjectId(_id)
        obj = super(BaseModelView, self).get_one(_id)
        if not obj:
            if isinstance(_id, ObjectId):
                return super(BaseModelView, self).get_one(str(_id))
            else:
                try:
                    return super(BaseModelView, self).get_one(ObjectId(_id))
                except:
                    pass
        return obj

    # Permissions

    """
        security levels in descending order:
        'private' - users need role to review resource, and another role to edit it's content
        'protected' - users without role can view, but not edit resource content
        'public' - everyone can edit resource content

        for all these security levels except 'public' users without role 'editor' are entitled to view content,
        but not edit it. Users with role 'editor' entitled to edit, obviously.
        User's with role 'manager' entitled to everything 'editors' are, as well as setting/removing roles
        for this resource.
    """

    security_level = 'public'

    resource_name = None  # must be defined for giving permissions (So required for all non-public resources)

    def is_accessible(self):  # Overriding flask permission method
        try:
            access.request_view(self)
            return True
        except Forbidden:
            return False

    def _handle_view(self, name, **kwargs):
        super(BaseModelView, self)._handle_view(name, **kwargs)
        if not hasattr(request, 'read_only'):
            access.request_edit(self)
            if request.read_only:
                flash("You don't have permission to edit %s. All changes will be discarded" % self.resource_name,
                      'error')

    @property
    def can_delete(self):
        return not request.read_only

    @property
    def can_import(self):
        return not request.read_only

    @property
    def can_export(self):
        return True  # by default all document could be exported

    @property
    def can_create(self):
        return not request.read_only

    @property
    def can_edit_permissions(self):
        return access.can_edit_permissions(self)
