#!/usr/bin/env python

import argparse
from urlparse import urljoin

import requests

def ok():
    return '0;Ok'

def warning(msg=None):
    return '1;%s' % (msg or 'Warning')

def failed(msg=None):
    return '2;%s' % (msg or 'Failed')

def check_downloads(response):
    data = response.json()
    if len(data) == 0:
        return failed("Empty response")
    for update in data:
        response = requests.head(update['download_url'])
        response.raise_for_status()
    return ok()

client_settings = {
    "nginx_url": "http://127.0.0.1",
    "checks": {
        "/api/v1/updates": {
            "params": {
                "locale": "ru_RU",
                "type": "yandex"
            },
            "method": "GET",
            "extra_check": check_downloads
        }
    },
    "http_headers": {
        "Host": "updater.mobile.yandex.net",
        "User-Agent": "com.yandex.launcher.updaterapp/1.6.ui.1009286 (Yandex YNDX-000SB; Android 8.1.0)",
        "X-YaUuid": "94270b66-0a1c-4aaf-bce9-5f0e2782a3ea"
    }
}

def perform_check(path, method, params, headers, extra_check):
    if not path.endswith('/'):
        path += '/'

    url = urljoin(client_settings['nginx_url'], path)

    try:
        response = requests.request(url=url, method=method, params=params, headers=headers)
        response.raise_for_status()
        if extra_check:
            return extra_check(response)
    except requests.HTTPError as ex:
        if ex.response.status_code == 400:
            return warning(str(ex))
        return failed("%s: %s" % (str(ex), url))
    except ValueError as ex:
        return failed(str(ex))

    return ok()


parser = argparse.ArgumentParser()
parser.add_argument('--check', type=str, dest='check')

def main():
    args = parser.parse_args()
    try:
        check = client_settings['checks'][args.check]
        headers = client_settings['http_headers']
        headers.update(check.get('http_headers', {}))
        return perform_check(args.check, check.get('method', 'GET'), check.get('params', {}), headers, check.get('extra_check'))
    except KeyError:
        return warning('Improperly configured, nothing to do')
    except Exception as e:
        return warning('Something wrong: {}'.format(e))


if __name__ == '__main__':
    print main()
