import logging
import json

from flask import Blueprint, jsonify, request, current_app

from yaphone.newpdater.src.admin import service, tvm

ROLE_NAME = 'роль'

logger = logging.getLogger(__name__)
api = Blueprint('idm', __name__, url_prefix='/updater')

ROLE_SLUG = 'role'
IDM_REQ_ID_HEADER = 'X-IDM-Request-Id'


@api.before_request
def check_tvm():
    header = request.headers.get(tvm.SERVICE_TICKET_HEADER)
    if header:
        parsed_ticket = tvm.client.parse_service_ticket(header)
        if parsed_ticket:
            src = str(parsed_ticket.src)
            expected_src = str(current_app.config['IDM_TVM_ID'])
            if src == expected_src:
                return
    return jsonify({
        'code': 401,
        'error': 'TVM auth failed'
    })


@api.before_request
def log_idm_request_id():
    header = request.headers.get(IDM_REQ_ID_HEADER)
    if header:
        logger.info('Got IDM request id %s', header)


# We should not raise any exception, return 200 always >_<
@api.errorhandler(Exception)
def handle_error(err):
    logger.exception('IDM exception')
    return jsonify({
        'code': 1,
        'error': 'Some error: %r' % err,
    })


@api.route('/info/')
def info():
    roles = service.get_roles_description()
    return jsonify({
        'code': 0,
        'roles': {
            'slug': ROLE_SLUG,
            'name': ROLE_NAME,
            'values': roles
        },
    })


def get_user_role():
    username = request.form['login']
    role_data = request.form['role']
    role_data = json.loads(role_data)
    role = role_data[ROLE_SLUG]
    return username, role


@api.route('/add-role/', methods=['POST'])
def add_role():
    username, role = get_user_role()
    service.add_role(username, role)
    return jsonify({
        'code': 0
    })


@api.route('/remove-role/', methods=['POST'])
def remove_role():
    username, role = get_user_role()
    service.delete_role(username, role)
    return jsonify({
        'code': 0
    })


@api.route('/get-all-roles/')
def get_all_roles():
    users = []
    for user in service.get_all_users():
        roles = [{ROLE_SLUG: role} for role in user.get_roles()]
        user_roles = {'login': user.username, 'roles': roles}
        users.append(user_roles)
    return jsonify({
        'code': 0,
        'users': users,
    })
