from tvm2 import TVM2

SERVICE_TICKET_HEADER = 'X-Ya-Service-Ticket'


class TVMServiceTicketError(ValueError):
    pass


class TVMUninitialized(Exception):
    pass


class TVM:
    def __init__(self, app=None):
        self._client = None
        if app is not None:
            self.init_app(app)

    @property
    def client(self):
        if self._client is None:
            raise TVMUninitialized('TVM call out of application context')
        return self._client

    def init_app(self, app):
        self._client = TVM2(
            client_id=app.config['TVM_CLIENT_ID'],
            secret=app.config.get('TVM_CLIENT_SECRET'),
            blackbox_client=app.config['BLACKBOX_CLIENT']
        )

    def add_service_ticket_header(self, client_id, headers=None):
        """
        Method patches headers dict, adding "X-Ya-Service-Ticket" header for proper client_id
        :param client_id: string
        :param headers: dict
        :return: dict
        """
        if headers is None:
            headers = {}

        tickets = self.client.get_service_tickets(client_id)
        ticket = tickets.get(client_id)

        if not ticket:
            raise TVMServiceTicketError('Empty TVM service-ticket for client_id: %s' % client_id)

        headers['X-Ya-Service-Ticket'] = ticket
        return headers

    def parse_service_ticket(self, service_ticket):
        return self.client.parse_service_ticket(service_ticket)


client = TVM()


def init_app(app):
    client.init_app(app)
