import logging
import urllib.parse
import weakref
from typing import Optional

import werkzeug.exceptions
import wtforms.validators
import yenv
from flask import abort, redirect, request, g, current_app
from flask_admin import Admin, AdminIndexView
from flask_admin.base import expose
from flask_admin.contrib.sqla import ModelView
from flask_admin.form import BaseForm
from flask_admin.model.template import LinkRowAction
from werkzeug import datastructures
from wtforms.fields import StringField

from yaphone.newpdater.src.admin import auth, models

logger = logging.getLogger(__name__)


def strip_whitespace(value: str) -> Optional[str]:
    if value is not None and hasattr(value, 'strip'):
        return value.strip()
    return value


# https://github.com/alphagov/digitalmarketplace-supplier-frontend/pull/297
class StripWhitespaceForm(BaseForm):
    class Meta:
        def bind_field(self, form, unbound_field, options):
            filters = unbound_field.kwargs.get('filters', [])
            filter_fields = (StringField,)
            if issubclass(unbound_field.field_class, filter_fields) and \
                    strip_whitespace not in filters:
                filters.append(strip_whitespace)
            bound = unbound_field.bind(form=form, filters=filters, **options)
            bound.get_form = weakref.ref(form)  # GC won't collect the form if we don't use a weakref
            return bound


class AuthMixin:
    access_roles = models.ROLES

    def get_user(self):
        if not hasattr(g, 'user'):
            try:
                g.user = auth.get_yateam_user()
            except auth.AuthError:
                raise werkzeug.exceptions.ServiceUnavailable('Auth service unavailable')
        return g.user

    def is_accessible(self):
        current_user = self.get_user()
        return current_user.is_authenticated and any(
            current_user.has_role(role) for role in self.access_roles
        )

    def is_visible(self):
        return self.is_accessible()

    def inaccessible_callback(self, name, **kwargs):
        user = self.get_user()
        if user.is_authenticated:
            return abort(403)
        else:
            url = urllib.parse.quote(request.url)
            return redirect(f'https://passport.yandex-team.ru/auth?retpath={url}')


class SplitChangeModelView(ModelView):

    def on_model_change(self, form, model, is_created):
        if is_created:
            logger.debug('call on_model_created')
            self.on_model_created(form, model)
        else:
            logger.debug('call on_model_edited')
            self.on_model_edited(form, model)
        super().on_model_change(form, model, is_created)

    def on_model_created(self, form, model):
        pass

    def on_model_edited(self, form, model):
        pass


class AdminModelView(AuthMixin, ModelView):
    form_base_class = StripWhitespaceForm


class DownloadUrlRowAction(LinkRowAction):
    def __init__(self, url_maker, icon_class='glyphicon glyphicon-download', title=None):
        self.url_maker = url_maker
        super().__init__(icon_class=icon_class, title=title, url=None)

    def render(self, context, row_id, row):
        self.url = self.url_maker(row)
        if self.url is not None:
            return super().render(context, row_id, row)
        return ''


# noinspection PyMethodMayBeStatic
class FileUploadView(SplitChangeModelView, AdminModelView):
    column_exclude_list = ('filename',)

    def validate_file_uploaded(self, form, is_created, field='file') -> datastructures.FileStorage:
        if not hasattr(form, field) or is_created and getattr(form, field).data is None:
            raise wtforms.validators.ValidationError('File %s is required on creation' % field)
        return getattr(form, field).data

    def get_optional_data(self, form, field):
        if hasattr(form, field):
            return getattr(form, field).data

    def should_overwrite(self, form) -> bool:
        return form.overwrite.data if hasattr(form, 'overwrite') else False


class IndexView(AuthMixin, AdminIndexView):

    @expose()
    def index(self):
        self._template_args['environ'] = yenv.type.upper()
        self._template_args['label_class'] = current_app.config.get('ADMIN_LABEL_STYLE')
        self._template_args['lc_admin_url'] = current_app.config.get('ADMIN_LOCALIZATION_URL')
        return super(IndexView, self).index()

    def is_accessible(self):
        return self.get_user().is_authenticated


admin_views = Admin(
    name='Updater admin',
    template_mode='bootstrap3',
    index_view=IndexView(name='Home', url='', template='admin/index.html'),
)
