from yaphone.localization import Translator, UserSpecificConfig, LocalizationUser, init_cache
from yaphone.newpdater.src.common.exceptions import ConfigurationError
from yaphone.newpdater.src.common.mongo_tools import make_mongo_uri
from yaphone.utils.parsers.parsers import UaParser, FingerprintParser, DEVICE_PHONE


class LocalizationService:
    server_project = ''
    client_project = ''
    translate_project = ''

    def configure(self, server_project_name, client_project_name, translate_project_name):
        self.server_project = server_project_name
        self.client_project = client_project_name
        self.translate_project = translate_project_name

    @staticmethod
    def normalize_device_type(ua_device_type):
        return ua_device_type.lower() if ua_device_type else DEVICE_PHONE

    # noinspection DuplicatedCode
    @staticmethod
    def create_user(uuid, locale=None, user_agent=None, serial_number=None, yphone_id=None, build_fingerprint=None,
                    mac_address_wifi=None, mac_address_ethernet=None):
        user = LocalizationUser(uuid=uuid)

        if locale is not None:
            user.language = locale

        if serial_number is not None:
            user.serial_number = serial_number

        if yphone_id is not None:
            user.yphone_id = yphone_id

        if mac_address_wifi is not None:
            user.mac_address_wifi = mac_address_wifi

        if mac_address_ethernet is not None:
            user.mac_address_ethernet = mac_address_ethernet

        if user_agent is None:
            return user

        build_fields = FingerprintParser.parse(build_fingerprint) if build_fingerprint else {}
        if build_fields:
            user.build_version = build_fields['build_number']
            user.os_version = build_fields['version_release']

        fields = UaParser.parse(user_agent)
        if fields:
            user.app_name = fields['app_name']
            user.app_version = fields['app_version_string']
            user.device_manufacturer = fields['device_manufacturer']
            user.device_model = fields['device_model']
            user.device_type = LocalizationService.normalize_device_type(fields['device_type'])
            user.os_name = fields['os_name']
            user.os_version = user.os_version or fields['os_version']

        return user

    def create_config(self, user, project):
        if project:
            return UserSpecificConfig(user, project)
        else:
            raise ConfigurationError(
                'project_name cannot be empty. '
                'Has init_app been called or is LOCALIZATION_PROJECT empty?'
            )

    def create_translator(self, user):
        if self.translate_project:
            return Translator(user, self.translate_project)
        else:
            raise ConfigurationError(
                'translate_name cannot be empty. '
                'Has init_app been called? '
                'Does LOCALIZATION_PROJECTS contain data for translate?'
            )


class ConfigFactory:

    def __init__(self, uuid, user_agent, serial_number=None, locale=None, build_fingerprint=None, mac_wifi=None,
                 mac_ethernet=None, **_):
        self._server_config = None
        self._client_config = None
        self._translator = None
        self.user = service.create_user(
            uuid=uuid,
            user_agent=user_agent,
            locale=locale,
            serial_number=serial_number,
            build_fingerprint=build_fingerprint,
            mac_address_wifi=mac_wifi,
            mac_address_ethernet=mac_ethernet
        )

    @property
    def server_config(self) -> UserSpecificConfig:
        if not self._server_config:
            self._server_config = service.create_config(self.user, service.server_project)
        return self._server_config

    @property
    def client_config(self) -> UserSpecificConfig:
        if not self._client_config:
            self._client_config = service.create_config(self.user, service.client_project)
        return self._client_config

    @property
    def translator(self) -> Translator:
        if not self._translator:
            self._translator = service.create_translator(self.user)
        return self._translator


service = LocalizationService()


# Called by src/core/app.py
def init_app(app):
    project_names = app.config['LOCALIZATION_PROJECTS']
    service.configure(
        project_names['server'],
        project_names['client'],
        project_names['translate']
    )

    init_cache(
        db_name=app.config['LOCALIZATION_MONGO_DBNAME'],
        mongo_uri=app.config.get('MONGO_URI', make_mongo_uri(app.config, 'LOCALIZATION_MONGO_DBNAME')),
        projects=app.config['LOCALIZATION_PROJECTS'].values()
    )
