import logging
from typing import BinaryIO, Optional

import boto3
from boto3.s3.transfer import TransferConfig
from botocore.errorfactory import ClientError
from werkzeug.exceptions import InternalServerError

logger = logging.getLogger(__name__)


class MockClient:
    bucket = []

    def upload_fileobj(self, Key, **_):
        self.bucket.append(Key)
        logger.debug('[upload_fileobj] Put a new "%s" object to S3', Key)
        return 1024

    def head_object(self, Key, **_):
        if Key in self.bucket:
            logger.debug('[head_object] There is such key "%s"', Key)
            return {'ContentLength': 1024}
        else:
            logger.debug('[head_object] Error: no such key "%s"', Key)
            raise ClientError({'Error': {'Code': 404}}, {})

    def delete_object(self, Key, **_):
        try:
            self.bucket.remove(Key)
            logger.debug('[delete_object] Delete an "%s" object from S3', Key)
        except ValueError:
            logger.debug('[delete_object] Cannot delete %s: nothing to delete', Key)


class ResultCallback:
    def __init__(self, result_size, on_finish=lambda success: None):
        self._size = result_size
        self._loaded = 0
        self._callback = on_finish

    def __call__(self, bytes_amount):
        self._loaded += bytes_amount
        percentage = round((self._loaded / self._size) * 100)
        if percentage == 100:
            self._callback(success=True)


class S3Metadata:
    md5_hash = None

    def __init__(self, data: dict):
        self._data = data

    @property
    def key(self):
        return self._data.get('Key')

    @property
    def size(self) -> int:
        return self._data.get('ContentLength')

    # Sometimes ETag is not MD5
    # @property
    # def md5_hash(self) -> str:
    #     return self._data.get('ETag').strip('"')[:32]


class S3Client:

    def __init__(self):
        self.main_host = ''
        self.public_host = ''
        self.bucket_name = ''
        self.url_template = ''
        self.client = None

    def configure(self, config):
        self.main_host = config['S3_MAIN_HOST']
        self.public_host = config['S3_PUBLIC_HOST']
        self.bucket_name = config['S3_BUCKET_NAME']
        self.url_template = config['S3_URL_TEMPLATE']
        if config.get('S3_MOCK_CLIENT', False):
            self.client = MockClient()
        else:
            self.client = boto3.session.Session(
                aws_access_key_id=config['S3_ACCESS_KEY_ID'],
                aws_secret_access_key=config['S3_ACCESS_SECRET_KEY'],
            ).client(
                service_name='s3',
                endpoint_url=f'https://{self.main_host}',
            )

    def upload_file(self, stream: BinaryIO, key: str,
                    filename: Optional[str] = None, content_type: Optional[str] = None):
        try:
            transfer_config = TransferConfig(
                max_io_queue=10,
            )
            extra_args = {}
            if filename:
                extra_args['ContentDisposition'] = f'attachment; filename="{filename}"'
            if content_type:
                extra_args['ContentType'] = content_type
            self.client.upload_fileobj(
                Bucket=self.bucket_name,
                Fileobj=stream,
                Key=key,
                ExtraArgs=extra_args,
                Config=transfer_config,
            )
        except ClientError:
            exception_reason = 'Could not upload file to storage'
            logger.exception(exception_reason)
            raise InternalServerError(exception_reason)

    def make_url(self, key: str) -> str:
        return self.url_template.format(
            bucket_name=self.bucket_name,
            public_host=self.public_host,
            main_host=self.main_host,
            key=key,
        )

    def delete_file(self, key: str):
        self.client.delete_object(Bucket=self.bucket_name, Key=key)

    def get_metadata(self, key: str) -> Optional[S3Metadata]:
        try:
            return S3Metadata(self.client.head_object(Bucket=self.bucket_name, Key=key))
        except ClientError:
            return None

    def get_file_size(self, key: str) -> Optional[int]:
        metadata = self.get_metadata(key)
        if metadata is None:
            return None
        logger.info('Metadata for key %s: %s', key, metadata)
        return metadata.size

    def file_exists(self, key: str) -> bool:
        return self.get_metadata(key) is not None

    def move_file(self, old_key: str, new_key: str):
        """
         Since boto3 can't actually move or rename files,
         this method copies an old file, and then removes it.
         """

        def on_moved(success):
            if success:
                self.delete_file(old_key)

        try:
            callback = ResultCallback(self.get_file_size(old_key), on_finish=on_moved)
            copy_from = dict(Bucket=self.bucket_name, Key=old_key)
            self.client.copy(copy_from, Bucket=self.bucket_name, Key=new_key, Callback=callback)
        except ClientError:
            on_moved(success=False)
