from abc import ABC, abstractmethod
from hashlib import sha1
from typing import Collection
import re

from flask_sqlalchemy import model

TAG_SIZE = 8
"""Size of an s3mds key tag"""


def generate_tag(*args):
    tag = '&'.join([str(arg) for arg in args])
    return sha1(tag.encode()).hexdigest()[:TAG_SIZE]


def whitespaces_to_underscore(entry):
    return re.sub(r'\s+', '_', entry)


class Key(ABC):
    """Class to create a unique key for an entity."""
    key: str
    tag: str

    entity = None
    """An object to take fields from."""
    key_format: str = None
    """A key format. E.g.: 'files/{name}/{tag}/{version}.'"""
    key_fields: Collection = None
    """Fields, which should be included to a key. Uses all fields by default."""
    tag_fields: Collection = None
    """Fields to generate a tag from. Uses all fields by default."""

    def __init__(self):
        self.key_fields = self.key_fields or self._get_all_field_names()
        self.tag_fields = self.tag_fields or self.key_fields
        fields = self._take_fields(self.key_fields)

        if self.is_tagged_key():
            self.tag = fields.pop('tag', None) or generate_tag(self._take_values(self.tag_fields))
            self.key_format = self.key_format or self._make_key_format()
            self.key = self.key_format.format(tag=self.tag, **fields)
            self.key = whitespaces_to_underscore(self.key)
        else:
            self.key = self.key_format.format(**fields)
            self.key = whitespaces_to_underscore(self.key)

    def is_tagged_key(self):
        return (self.key_format and '{tag}' in self.key_format) or (not self.key_format)

    def _make_key_format(self):
        """
        Makes a key format using self.key_fields and a tag.
        This method is used if a user doesn't define a custom self.key_format.
        """
        fields = ['{%s}' % field for field in self.key_fields]
        if 'tag' not in self.key_fields:
            fields.append(self.tag)
        return '/'.join(fields)

    def __str__(self):
        return repr(self)

    def __repr__(self):
        return self.key

    @abstractmethod
    def _get_all_field_names(self):
        raise NotImplementedError

    def _take_fields(self, field_names: Collection) -> dict:
        return {key: getattr(self.entity, key) for key in field_names}

    def _take_values(self, field_names: Collection) -> list:
        return [getattr(self.entity, key) for key in field_names]


class GenericKey(Key, ABC):
    def __init__(self, entity=None, key_format=None, key_fields=None, tag_fields=None):
        self.entity = entity or self.entity
        self.key_format = key_format or self.key_format
        self.key_fields = key_fields or self.key_fields
        self.tag_fields = tag_fields or self.tag_fields
        super().__init__()


class AlchemyKey(GenericKey):

    def _get_all_field_names(self):
        return self.entity.__table__.c.keys()


class DictKey(GenericKey):

    def _take_fields(self, field_names: Collection) -> dict:
        return {key: self.entity[key] for key in field_names}

    def _take_values(self, field_names: Collection) -> list:
        return [self.entity[key] for key in field_names]

    def _get_all_field_names(self):
        return self.entity.keys()


class KeyMaker:
    def __init__(self, key_format=None, key_fields=None, tag_fields=None):
        self.key_format = key_format
        self.key_fields = key_fields
        self.tag_fields = tag_fields

    def make_key(self, entity) -> str:
        return create_key(entity, self.key_format, self.key_fields, self.tag_fields)


def create_key(entity, key_format=None, key_fields=None, tag_fields=None):
    if isinstance(entity, dict):
        key_type = DictKey
    elif isinstance(entity, model.Model):
        key_type = AlchemyKey
    else:
        raise TypeError(f'Unsupported key_type {type(entity)}')

    return str(key_type(entity, key_format, key_fields, tag_fields))
