import hashlib
import logging
from typing import BinaryIO, Optional

from werkzeug.exceptions import BadRequest, InternalServerError

from yaphone.newpdater.src.common.s3mds import s3client

logger = logging.getLogger(__name__)


class FileLoaderService:

    def __init__(self, client: s3client.S3Client):
        self.client = client

    def make_url(self, key: str):
        return self.client.make_url(key)

    def check_file_exists(self, key: str, overwrite: bool = False, exception=BadRequest):
        if not overwrite and self.client.file_exists(key):
            logger.debug('check_file_exists: file already exists %s', key)
            raise exception('File already exists')

    @staticmethod
    def _calculate_md5(filestream: BinaryIO, blocksize=65536) -> str:
        hasher = hashlib.md5()
        block = filestream.read(blocksize)
        while len(block) > 0:
            hasher.update(block)
            block = filestream.read(blocksize)
        return hasher.hexdigest()

    def upload_file(self, key: str, filestream: BinaryIO,
                    filename: Optional[str] = None, content_type: Optional[str] = None) -> s3client.S3Metadata:
        md5_hash = None
        if (hasattr(filestream, 'seekable') and filestream.seekable()) or hasattr(filestream, 'seek'):
            md5_hash = self._calculate_md5(filestream)
            filestream.seek(0)
        else:
            logger.warning('Filestream is not seekable, MD5 will not be calculated')
        self.client.upload_file(stream=filestream, key=key, filename=filename, content_type=content_type)
        metadata = self.client.get_metadata(key)
        if metadata is None:
            logger.error('Could not upload file to %s', key)
            raise InternalServerError('Could not upload file to storage')
        metadata.md5_hash = md5_hash
        return metadata

    def move_file(self, old_key: str, new_key: str, overwrite: bool = False):
        old_exists = self.client.file_exists(old_key)
        new_exists = self.client.file_exists(new_key)
        if not old_exists:
            raise BadRequest(f'Cannot move: no such key {old_key}')
        if new_exists and not overwrite:
            raise BadRequest(f'Cannot move: {new_key} already exists')
        if old_key != new_key:
            self.client.move_file(old_key, new_key)
            logger.info('Successfully moved %s to %s', old_key, new_key)
        else:
            logger.info('Trying to move %s to the same place, ignoring', old_key)

    def delete_file(self, key: str):
        self.client.delete_file(key)
