import logging
from typing import Type

from marshmallow import Schema, ValidationError
from marshmallow.fields import String, UUID
from marshmallow.utils import missing
from marshmallow.validate import Regexp
from werkzeug.exceptions import BadRequest
from yaphone.utils.parsers import parsers

BUILD_FIELD = 'Build-Fingerprint'
MAC_WIFI_FIELD = 'X-Wifi-Mac'
MAC_ETHERNET_FIELD = 'X-Ethernet-Mac'

logger = logging.getLogger(__name__)


class HeadersValidator(Schema):
    uuid = UUID(required=True, load_from='X-Yauuid')
    user_agent = String(
        required=True,
        load_from='User-Agent',
        validate=Regexp(parsers.USER_AGENT_REGEXP)
    )


# TODO: shouldn't build_fingerprint be required?
class DeviceHeadersValidator(HeadersValidator):
    build_fingerprint = String(
        required=False,
        load_from=BUILD_FIELD,
        validate=Regexp(parsers.BUILD_FINGERPRINT_REGEXP)
    )
    mac_wifi = String(
        required=False,
        load_from=MAC_WIFI_FIELD
    )
    mac_ethernet = String(
        required=False,
        load_from=MAC_ETHERNET_FIELD
    )


def validate(schema: Type[Schema], request_data, unpack=False):
    try:
        data = schema(strict=True).load(request_data).data
        if unpack:
            unpacked = to_dict(schema)
            unpacked.update(data)
            return unpacked.values()
        else:
            return data
    except ValidationError as ex:
        logger.exception(str(ex))
        raise BadRequest(ex)


def validate_many(schema: Type[Schema], request_data: list):
    try:
        return schema(strict=True).load(request_data, many=True).data
    except ValidationError as ex:
        logger.exception(str(ex))
        raise BadRequest(ex)


def to_dict(schema: Type[Schema]):
    """
    Transforms a marshmallow.Schema class
    to a dict of Schema's fields and their default values.
    :param schema: Class to take fields and default values from
    :return: dict of fields and default values, or Nones
    """
    schema_dict = {}
    fields = schema._declared_fields
    for k, v in fields.items():
        schema_dict[k] = v.default if v.default is not missing else None
    return schema_dict
