import sys
import os
import logging
import re
import jinja2
import library.python.resource as lpr
from flask import Flask
from werkzeug.middleware.proxy_fix import ProxyFix
from werkzeug.routing import BaseConverter

from yaphone.newpdater.src.admin import tvm
from yaphone.newpdater.src.common import databases, localization, s3mds
from yaphone.newpdater.src.core import log_context, routes, settings, sentry


logger = logging.getLogger(__name__)


class RegexpConverter(BaseConverter):
    def __init__(self, url_map, *args):
        super().__init__(url_map)
        self.regex = args[0]


def get_updater_binary_path():
    return sys.executable


def get_newest_available_migration():
    migrations_folder = os.path.join(os.path.abspath(os.path.dirname(get_updater_binary_path())), 'migrations/versions')
    try:
        migration_files = list(os.listdir(migrations_folder))
        if not migration_files:
            return None
        pattern = re.compile('^\\d{4}.*')
        migration_files = sorted([f for f in migration_files if pattern.match(f)], reverse=True)
        if not migration_files:
            return None
        return migration_files[0]
    except FileNotFoundError:
        logger.exception('Can not find migrations')
        return None


def create_app():
    app = Flask(__name__)
    app.config.from_object(settings.load_current_config())
    app.config['NEWEST_MIGRATION'] = get_newest_available_migration()
    app.wsgi_app = ProxyFix(app.wsgi_app)
    app.url_map.converters['re'] = RegexpConverter
    routes.init_app(app)
    databases.init_app(app)
    localization.init_app(app)
    s3mds.init_app(app)
    tvm.init_app(app)
    sentry.init_app(app)
    app.wsgi_app = log_context.YLogMiddleware(app.wsgi_app)
    app.jinja_loader = jinja2.DictLoader(
        {
            key: value.decode('utf-8')
            for key, value in
            lpr.iteritems(
                prefix='/templates/',
                strip_prefix=True,
            )
        }
    )
    return app
