import logging
import os
from logging.config import dictConfig

import yenv


UNIFIED_AGENT_LOGS_URI = os.getenv('UNIFIED_AGENT_LOGS_URI')


def get_log_level():
    log_level = os.getenv('LOGGING_LEVEL')
    if log_level not in ('CRITICAL', 'FATAL', 'ERROR', 'WARNING', 'WARN', 'INFO', 'DEBUG'):
        log_level = yenv.choose_key_by_type({
            'production': logging.INFO,
            'testing': logging.DEBUG,
            'stress': logging.INFO,
        }, fallback=True)
    return log_level


def get_log_handlers():
    handlers = {
        'console': {
            'class': 'logging.StreamHandler',
            'formatter': 'default',
        },
        'qloud': {
            'class': 'logging.StreamHandler',
            'formatter': 'json',
        },
    }

    if UNIFIED_AGENT_LOGS_URI:
        handlers['logbroker'] = {
            'class': 'logbroker.unified_agent.client.python.UnifiedAgentYdHandler',
            'logger_name': 'stdout',
            'formatter': 'json',
            'uri': UNIFIED_AGENT_LOGS_URI,
        }

    return handlers


def get_log_handlers_list():
    handlers = yenv.choose_key_by_type({
        'production': ['qloud'],
        'stress': ['qloud'],
        'development': ['console'],
    }, fallback=True)

    if UNIFIED_AGENT_LOGS_URI:
        handlers.append('logbroker')

    return handlers


LOGGING_CONFIG = {
    'version': 1,
    'disable_existing_loggers': False,
    'formatters': {
        'default': {
            'format': '[%(asctime)s] [%(process)d] [%(levelname)s] [%(name)s] %(message)s',
            'datefmt': '%Y-%m-%d %H:%M:%S',
            'class': 'logging.Formatter',
        },
        'raw': {
            'format': '%(message)s',
            'datefmt': '',
            'class': 'logging.Formatter',
        },
        'json': {
            '()': 'ylog.format.QloudJsonFormatter',
        },
    },
    'handlers': get_log_handlers(),
    'loggers': {
        '': {
            'level': get_log_level(),
            'handlers': get_log_handlers_list(),
            'propagate': True,
        },
        'requests': {
            'level': 'WARN'
        },
        'gunicorn.access': {
            'handlers': get_log_handlers_list(),
        },
        'gunicorn.error': {
            'handlers': get_log_handlers_list(),
        },
    },
}


def init():
    dictConfig(LOGGING_CONFIG)
