import os
import logging

import yenv
from tvmauth import BlackboxTvmId as BlackboxClientId

SQLALCHEMY_URI_TEMPLATE = ('postgresql://{username}:{password}@{host}:{port}/{db_name}?'
                           'sslmode=verify-full&target_session_attrs=read-write')

os.environ.setdefault('AWS_CA_BUNDLE', '/etc/ssl/certs/yandex-ca.pem')
os.environ.setdefault('PGSSLROOTCERT', '/etc/ssl/certs/YandexInternalRootCA.pem')
os.environ.setdefault('SENTRY_DSN', yenv.choose_key_by_type({
    'production': 'https://07b2b2ab2d1249eea88e117531855ec6'
                  '@sentry.iddqd.yandex.net/1075?verify_ssl=0',
    'prestable': 'https://41a9803df96e43b1ad1983da8a1077d0'
                 '@sentry-test-proxy.t.yandex.net/583?verify_ssl=0',
    'stress': '',
    'development': '',
}, fallback=True))


class AppConfig:
    ADMIN_URL = '/updater/admin'
    CSRF_ENABLED = True
    SECRET_KEY = os.environ.get('SECRET_KEY', '______')

    SQLALCHEMY_ECHO = False
    SQLALCHEMY_DATABASE_URI = ''
    SQLALCHEMY_TRACK_MODIFICATIONS = False

    # TODO: There is no SQLALCHEMY_ENGINE_OPTIONS in flask_sqlalchemy 2.3.
    #   Remove *_POOL_* options after migrating to 2.4+ and use SQLALCHEMY_ENGINE_OPTIONS instead.
    SQLALCHEMY_POOL_SIZE = os.environ.get('PG_POOL_SIZE', 10)
    SQLALCHEMY_MAX_OVERFLOW = 1
    SQLALCHEMY_POOL_RECYCLE = 30 * 60  # 30 min
    SQLALCHEMY_POOL_PRE_PING = True  # Not an official property, see CONTRIB-162
    SQLALCHEMY_ENGINE_OPTIONS = {
        'pool_size': SQLALCHEMY_POOL_SIZE,
        'max_overflow': SQLALCHEMY_MAX_OVERFLOW,
        'pool_recycle': SQLALCHEMY_POOL_RECYCLE,
        'pool_pre_ping': SQLALCHEMY_POOL_PRE_PING,
    }

    S3_MAIN_HOST = 's3.mds.yandex.net'
    S3_PUBLIC_HOST = 's3.yandex.net'
    S3_BUCKET_NAME = os.environ.get('S3_BUCKET_NAME')
    S3_URL_TEMPLATE = 'https://{bucket_name}.{public_host}/{key}'
    S3_ACCESS_KEY_ID = os.environ.get('S3_MDS_ACCESS_KEY_ID')
    S3_ACCESS_SECRET_KEY = os.environ.get('S3_MDS_ACCESS_SECRET_KEY')

    MONGO_PORT = 27017
    MONGO_HOST = ['localhost']
    MONGO_DB_NAME = os.environ.get('MONGO_DB_NAME')
    MONGO_USERNAME = os.environ.get('MONGO_USER')
    MONGO_PASSWORD = os.environ.get('MONGO_PASSWORD')

    LOCALIZATION_PROJECTS = {
        'server': 'updater',
        'client': 'updater_client_settings',
        'translate': 'launcher_translations',
    }
    LOCALIZATION_MONGO_DBNAME = os.environ.get('LOCALIZATION_DB_NAME')

    BLACKBOX_URL = 'https://blackbox.yandex-team.ru/blackbox'
    BLACKBOX_CLIENT = BlackboxClientId.ProdYateam
    IDM_TVM_ID = 2001600
    TVM_CLIENT_SECRET = os.environ.get('TVM_CLIENT_SECRET')

    SENTRY_DSN = os.getenv('SENTRY_DSN')
    SENTRY_ENVIRONMENT = yenv.type
    SENTRY_LEVEL = logging.WARNING


class ProductionConfig(AppConfig):
    SQLALCHEMY_DATABASE_URI = SQLALCHEMY_URI_TEMPLATE.format(
        db_name=os.environ.get('UPDATER_DB_NAME'),
        username=os.environ.get('UPDATER_DB_USER'),
        password=os.environ.get('UPDATER_DB_PASSWORD'),
        port='6432',
        host=os.environ.get('UPDATER_DB_HOSTS', ','.join([
            'vla-qcrac3eg8pb89zyp.db.yandex.net',
            'sas-7n8m65xxt8127g12.db.yandex.net',
            'sas-te5euk8ialm7i6tr.db.yandex.net',
            'vla-3kw64wvxvlhxlulm.db.yandex.net',
        ])),
    )

    MONGO_PORT = 27018
    MONGO_HOST = [
        'vla-8h8moxom5mcqvhxq.db.yandex.net',
        'sas-350atkewrtqdqznw.db.yandex.net',
        'iva-g4rokm43mg6rjizc.db.yandex.net',
    ]
    TVM_CLIENT_ID = 2021582

    ADMIN_LOCALIZATION_URL = 'https://localization-admin.phone.yandex.net/'
    ADMIN_LABEL_STYLE = 'label-primary'  # the blue one


class InfraConfig(AppConfig):
    ADMIN_URL = '/updater-infra/admin'

    # TODO: move to prod cluster after migration to infra hostname
    SQLALCHEMY_DATABASE_URI = SQLALCHEMY_URI_TEMPLATE.format(
        db_name=os.environ.get('UPDATER_DB_NAME'),
        username=os.environ.get('UPDATER_DB_USER'),
        password=os.environ.get('UPDATER_DB_PASSWORD'),
        port='6432',
        host=os.environ.get('UPDATER_DB_HOSTS', ','.join([
            'sas-nolx33gislpa0bzw.db.yandex.net',
            'vla-woyki56djlcyt7jd.db.yandex.net',
        ])),
    )

    MONGO_PORT = 27018
    MONGO_HOST = [
        'sas-m6ed1vwh01lxm0td.db.yandex.net',
        'vla-5xqt1wmnesk3vqto.db.yandex.net',
    ]
    TVM_CLIENT_ID = 2021584

    SENTRY_ENVIRONMENT = 'infra'

    ADMIN_LOCALIZATION_URL = 'https://localization-admin.tst.phone.yandex.net/'
    ADMIN_LABEL_STYLE = 'label-success'  # the green one


class TestingConfig(AppConfig):
    SQLALCHEMY_DATABASE_URI = SQLALCHEMY_URI_TEMPLATE.format(
        db_name=os.environ.get('UPDATER_DB_NAME'),
        username=os.environ.get('UPDATER_DB_USER'),
        password=os.environ.get('UPDATER_DB_PASSWORD'),
        port='6432',
        host=os.environ.get('UPDATER_DB_HOSTS', ','.join([
            'sas-nolx33gislpa0bzw.db.yandex.net',
            'vla-woyki56djlcyt7jd.db.yandex.net',
        ])),
    )

    S3_URL_TEMPLATE = 'https://{main_host}/{bucket_name}/{key}'

    MONGO_PORT = 27018
    MONGO_HOST = [
        'sas-m6ed1vwh01lxm0td.db.yandex.net',
        'vla-5xqt1wmnesk3vqto.db.yandex.net',
    ]
    TVM_CLIENT_ID = 2021580

    ADMIN_LOCALIZATION_URL = 'https://localization-admin.tst.phone.yandex.net/'
    ADMIN_LABEL_STYLE = 'label-success'  # the green one


class DevelopmentConfig(TestingConfig):
    """
    For configurations, specific to a development environment.
    If you don't need anything from TestingConfig at all,
    just replace it with AppConfig and add your own configurations.
    """
    SQLALCHEMY_ECHO = True
    SQLALCHEMY_DATABASE_URI = os.environ.get('UPDATER_DB_URI', 'postgresql://localhost/pythonbase')

    ADMIN_LOCALIZATION_URL = 'https://localization-admin.tst.phone.yandex.net/'
    ADMIN_LABEL_STYLE = 'label-default'  # the grey one


class LocalTestConfig(AppConfig):
    TESTING = True
    SQLALCHEMY_ECHO = True
    SQLALCHEMY_DATABASE_URI = 'sqlite://'

    MONGO_URI = 'mongomock://localhost'

    S3_MOCK_CLIENT = True
    S3_BUCKET_NAME = 'test_bucket'
    S3_PUBLIC_HOST = 'test_host'

    SQLALCHEMY_POOL_SIZE = None
    SQLALCHEMY_MAX_OVERFLOW = None
    SQLALCHEMY_ENGINE_OPTIONS = {}


def load_current_config():
    return yenv.choose_key_by_type({
        'production': ProductionConfig,
        'prestable': InfraConfig,
        'testing': TestingConfig,
        'development': DevelopmentConfig,
    }, fallback=True)
