from urllib.parse import urljoin

import pytest
from flask import Flask

from .fixtures import (
    NORMAL_DEVICE_ID, CHAOTIC_CASES_DEVICE_ID,
    NORMAL_EXISTING_DEVICE, CHAOTIC_CASES_DEVICE,
    NON_EXISTENT_DEVICE, NO_UPDATES_DEVICE,
    get_yandex_tv_app, get_yandex_services_app,
    get_bluetooth_app, get_updater_app, get_video_app, get_messaging_app,
    MISSING_VERSIONS_DEVICE, NAME_ONLY_DEVICE, MISSING_INFO_DEVICE,
)
from yaphone.newpdater.src.common import databases, s3mds
from yaphone.newpdater.src.core import settings, routes
from yaphone.newpdater.src.core.app import RegexpConverter
from yaphone.newpdater.src.prebuilt import models


def prepare_data_for_db(firmware_id, data):
    data = data.copy()
    data.update({'device_firmware_id': firmware_id})
    return data


def prepare_device_for_db(id_, data):
    data = data.copy()
    data.update({'id': id_})
    return data


def lower_values(data):
    return {k: v.lower() for k, v in data.items()}


def add_firmware(db, id_, firmware):
    normal_device = prepare_device_for_db(id_, firmware)
    normal_device = models.DeviceFirmware(**normal_device)
    db.session.add(normal_device)


def add_app(db, firmware_id, app):
    app = prepare_data_for_db(firmware_id, app)
    app = models.PrebuiltApp(**app)
    db.session.add(app)


def fill_database(db):
    add_firmware(db, NORMAL_DEVICE_ID, NORMAL_EXISTING_DEVICE)
    add_firmware(db, CHAOTIC_CASES_DEVICE_ID, CHAOTIC_CASES_DEVICE)

    add_app(db, NORMAL_DEVICE_ID, get_yandex_tv_app())
    add_app(db, NORMAL_DEVICE_ID, get_yandex_services_app())
    add_app(db, CHAOTIC_CASES_DEVICE_ID, get_bluetooth_app())
    add_app(db, CHAOTIC_CASES_DEVICE_ID, get_updater_app())
    add_app(db, CHAOTIC_CASES_DEVICE_ID, get_video_app())
    add_app(db, CHAOTIC_CASES_DEVICE_ID, get_messaging_app())


@pytest.fixture
def s3_client():
    app = Flask(__name__)
    app.config.from_object(settings.LocalTestConfig)
    app.url_map.converters['re'] = RegexpConverter
    routes.init_app(app)
    databases.init_app(app)
    s3mds.init_app(app)
    with app.app_context():
        db = databases.updater_db
        db.create_all()
        fill_database(db)
        db.session.commit()
        yield app.test_client()
        db.drop_all()


def add_s3_urls(firmware, apps):
    url_prefix = 'https://test_bucket.test_host/'
    for app in apps:
        key = f'prebuilt/{firmware["device"]}/{firmware["target"]}/{app["filename"]}'
        app['url'] = urljoin(url_prefix, key)
    return apps


DEVICE_FIRMWARES_WITH_RESULTS = [
    (NORMAL_EXISTING_DEVICE, [get_yandex_tv_app(), get_yandex_services_app()]),
    (CHAOTIC_CASES_DEVICE, [get_bluetooth_app(), get_updater_app(), get_video_app(), get_messaging_app()]),
]

MISSING_FIELDS_DEVICES = [
    MISSING_VERSIONS_DEVICE, MISSING_INFO_DEVICE, NAME_ONLY_DEVICE
]


class TestPrebuilt:
    url = '/api/v2/prebuilt'

    @pytest.mark.parametrize('firmware, expected', DEVICE_FIRMWARES_WITH_RESULTS)
    def test_prebuilt_succeeds(self, s3_client, firmware, expected):
        response = s3_client.get(self.url, query_string=firmware)

        assert response.json == add_s3_urls(firmware, expected)
        assert response.status_code == 200

    @pytest.mark.parametrize('device', [NON_EXISTENT_DEVICE, NO_UPDATES_DEVICE])
    def test_get_prebuilt_raises_no_such_device(self, s3_client, device):
        response = s3_client.get(self.url, query_string=device)

        assert response.status_code == 404

    @pytest.mark.parametrize('device', MISSING_FIELDS_DEVICES)
    def test_get_prebuilt_missing_fields_raises_bad_request(self, s3_client, device):
        response = s3_client.get(self.url, query_string=device)

        assert response.status_code == 400
