import logging

from flask import request, session
from flask.views import MethodView
from marshmallow import Schema, fields
from wtforms import BooleanField, ValidationError
from werkzeug.exceptions import BadRequest

from yaphone.newpdater.src.admin.fields import ApkFileField
from yaphone.newpdater.src.admin.views.base import AdminModelView, FileUploadView
from yaphone.newpdater.src.common import serialization, validation, views
from yaphone.newpdater.src.prebuilt import loader, repository, serializers, service

logger = logging.getLogger(__name__)


class PrebuiltAdminView(FileUploadView):
    form_extra_fields = dict(
        overwrite=BooleanField(default=False, label='Overwrite file'),
        file=ApkFileField(),
    )

    form_excluded_columns = ('filename', 'updated_at')

    column_list = ('local_name', 'module_name', 'privileged', 'updated_at')

    column_searchable_list = ('local_name', 'module_name', 'privileged', 'updated_at')

    column_labels = dict(
        local_name='LOCAL_PACKAGE_NAME',
        module_name='LOCAL_MODULE',
        module_class='LOCAL_MODULE_CLASS',
        certificate='LOCAL_CERTIFICATE',
        privileged='LOCAL_PRIVILEGED_MODULE',
        owner='LOCAL_MODULE_OWNER',
        platform_apis='LOCAL_PRIVATE_PLATFORM_APIS',
        overrides_packages='LOCAL_OVERRIDES_PACKAGES',
        module_tags='LOCAL_MODULE_TAGS',
    )

    form_widget_args = dict(
        filename=dict(readonly=True),
        updated_at=dict(readonly=True),
    )

    def on_form_prefill(self, form, id):
        device = serialization.serialize(serializers.DeviceSerializer, form.device_firmware.data)
        device['filename'] = repository.find_filename_by_id(id)
        session['device'] = device

    def on_model_created(self, form, model):
        filedata = self.validate_file_uploaded(form, True)
        if filedata:
            model.filename = filedata.filename
            overwrite = self.should_overwrite(form)
            key = self.make_new_key(model.device_firmware, model)
            loader.check_file_exists(key, overwrite, exception=ValidationError)
            metadata = loader.upload_file(key, filedata.stream)
            model.size = metadata.size

    def on_model_edited(self, form, model):
        if 'device' not in session:
            # Either there is an issue in error, or we launched a server being on the page "edit".
            raise BadRequest('Cannot obtain "device" object from session.')

        filedata = self.validate_file_uploaded(form, False)
        device = model.device_firmware
        overwrite = self.should_overwrite(form)
        old_key = service.key_maker.make_key(session['device'])
        new_key = self.make_new_key(device, model)
        loader.check_file_exists(new_key, overwrite, exception=ValidationError)
        if filedata:
            model.filename = filedata.filename
            new_key = self.make_key(device, model)
            metadata = loader.upload_file(new_key, filedata.stream)
            model.size = metadata.size
            loader.delete_file(old_key)
        else:
            loader.move_file(old_key, new_key, overwrite)

    # noinspection PyMethodMayBeStatic
    def make_new_key(self, device, model):
        # TODO: don't take a key_maker from service
        return service.key_maker.make_key({
            'device': device.device,
            'target': device.target,
            'filename': model.filename,
        })


class DeviceAdminView(AdminModelView):
    column_list = ('device', 'target', 'os_version', 'build_version')

    form_excluded_columns = ('app',)


class PrebuiltValidator(Schema):
    device = fields.String(required=True)
    target = fields.String(required=True)
    os_version = fields.String(required=True)
    build_version = fields.String(required=True)


class PrebuiltView(MethodView):
    def get(self):
        firmware = validation.validate(PrebuiltValidator, request.args)
        apps = service.get_prebuilt_apps(**firmware)
        return views.JsonResponse(apps)
