import mock
from flask import Flask

from yaphone.newpdater.src.common import localization
from yaphone.newpdater.src.core import settings, routes
from yaphone.newpdater.src.core.app import RegexpConverter
from .mocks import UserSpecificConfigMock, init_app


UPDATER_UA = 'com.yandex.launcher.updaterapp/1.0.0.214 (Fly FS509; Android 6.0)'
DEFAULT_TEST_UUID = '123456789012345678901234567890ab'


@mock.patch('yaphone.newpdater.src.common.localization.UserSpecificConfig', UserSpecificConfigMock)
class TestSettingsApi:
    uri = 'api/v2/settings'

    def setup_class(self):
        localization.init_app = init_app
        app = Flask(__name__)
        app.config.from_object(settings.LocalTestConfig)
        app.url_map.converters['re'] = RegexpConverter
        routes.init_app(app)
        localization.init_app(app)

        self.client = app.test_client()

    def request_and_check(self, headers, reference_status_code, extended_check=True):
        response = self.client.get(self.uri, headers=headers)
        assert response.status_code == reference_status_code, response.data

        if extended_check:
            obj = response.get_json()
            assert isinstance(obj, list)

            items = UserSpecificConfigMock().get_all_enabled_items()
            assert len(obj) == len(items)

    def test_settings_with_valid_headers(self):
        headers = {'X_YAUUID': DEFAULT_TEST_UUID, 'USER_AGENT': UPDATER_UA}
        self.request_and_check(headers, 200, True)

    def test_settings_with_invalid_uuid(self):
        headers = {'X_YAUUID': 'not uuid', 'USER_AGENT': UPDATER_UA}
        self.request_and_check(headers, 400, False)

    def test_settings_with_invalid_user_agent(self):
        headers = {'X_YAUUID': DEFAULT_TEST_UUID, 'USER_AGENT': 'Mozilla FF/2.1'}
        self.request_and_check(headers, 400, False)

    def test_settings_with_valid_optional_headers(self):
        headers = {
            'X_YAUUID': DEFAULT_TEST_UUID,
            'USER_AGENT': UPDATER_UA,
            'YPHONE-ID': '12345',
            'SERIAL': '12345'
        }
        self.request_and_check(headers, 200, True)
