import logging
import datetime
from functools import partial
from flask_admin.form import rules
from wtforms import fields
from wtforms.ext.sqlalchemy.fields import QuerySelectField

from yaphone.newpdater.src.admin.fields import ApkFileField, PictureField, DocumentField, ReadOnlyStringField
from yaphone.newpdater.src.admin.views.base import FileUploadView, DownloadUrlRowAction, AdminModelView
from yaphone.newpdater.src.store import service, loader

logger = logging.getLogger(__name__)


class ScreenshotView(FileUploadView):
    column_list = ('id', 'app_id', 'preview_url')

    column_extra_row_actions = [
        DownloadUrlRowAction(lambda obj: obj.url),
    ]

    form_widget_args = dict(
        s3key=dict(disabled=True),
        url=dict(disabled=True),
        preview_url=dict(disabled=True),
    )
    form_extra_fields = dict(
        screenshot=PictureField(),
    )

    form_create_rules = [
        rules.FieldSet(('screenshot', 'store_app')),
    ]

    def on_model_change(self, form, model, is_created):
        screenshot_data = self.validate_file_uploaded(form, is_created, field='screenshot')
        if screenshot_data:
            model.s3key = service.generate_img_s3key(model.store_app.package_name)
            loader.upload_file(model.s3key, screenshot_data.stream, screenshot_data.filename)

    def on_model_delete(self, model):
        if model.s3key:
            loader.delete_file(key=model.s3key)


class StoreAppView(FileUploadView):
    column_extra_row_actions = [
        DownloadUrlRowAction(lambda model: model.url),
    ]

    column_list = (
        'enabled',
        'app_name',
        'category_id',
        'package_name',
        'version_name',
        'installs_num',
        'updated_at',
        '_uploaded_at',
    )

    form_columns = (
        'app_name', 'app_name', 'category_id', 'description',
        'package_name', 'version_name', 'version_code', 'installs_num',
        'company_name', 'restriction_age', 'updated_at', '_uploaded_at',
        'url', 'file',
        'banner_url', 'banner',
        'eula_url', 'eula',
        'privacy_url', 'privacy',
        'enabled', 'targeting_string',
    )

    form_choices = {
        'targeting_string': [
            ('only_module', 'показывать только на модуле'),
            ('not_rt2871_hikeen', 'не показывать на rt2871_hikeen'),
            ('not_rt2871_not_cvte6681', 'не показывать на rt2871 и cvte6681'),
        ],
    }

    form_widget_args = dict(
        size=dict(disabled=True),
        installs_num=dict(disabled=True),
        updated_at=dict(disabled=True),
        url=dict(disabled=True),
        banner_url=dict(disabled=True),
        eula_url=dict(disabled=True),
        privacy_url=dict(disabled=True),
    )

    form_extra_fields = dict(
        description=fields.TextAreaField(),
        category_id=QuerySelectField(
            label='Category',
            query_factory=partial(service.get_categories),
            get_label='id',
            get_pk=lambda x: x.id
        ),
        file=ApkFileField(),
        banner=PictureField(),
        eula=DocumentField(label='EULA'),
        privacy=DocumentField(label='Privacy Policy'),
        url=ReadOnlyStringField(),
        banner_url=ReadOnlyStringField(),
        eula_url=ReadOnlyStringField(),
        privacy_url=ReadOnlyStringField(),
    )

    def on_model_change(self, form, model, is_created):
        apk_data = self.validate_file_uploaded(form, is_created)
        banner_data = self.validate_file_uploaded(form, is_created, field='banner')
        eula_data = self.get_optional_data(form, field='eula')
        privacy_data = self.get_optional_data(form, field='privacy')
        model.category_id = model.category_id.id
        if apk_data:
            model.s3key = service.generate_app_s3key()
            metadata = loader.upload_file(model.s3key, apk_data.stream, apk_data.filename)
            model.size = metadata.size
            model._uploaded_at = datetime.datetime.utcnow()
        if banner_data:
            model.banner_s3key = service.generate_img_s3key(model.package_name)
            loader.upload_file(model.banner_s3key, banner_data.stream, banner_data.filename)
        if eula_data:
            model.eula_s3key = service.generate_doc_s3key(model.package_name)
            loader.upload_file(model.eula_s3key, eula_data.stream, content_type='text/html')
        if privacy_data:
            model.privacy_s3key = service.generate_doc_s3key(model.package_name)
            loader.upload_file(model.privacy_s3key, privacy_data.stream, content_type='text/html')

    def on_model_delete(self, model):
        for key_field in ('s3key', 'banner_s3key', 'eula_s3key', 'privacy_s3key'):
            key = getattr(model, key_field)
            if key:
                loader.delete_file(key=key)


class CategoryView(AdminModelView):
    column_list = ('id', 'name')

    form_create_rules = [
        rules.FieldSet(('id', 'name')),
    ]
    form_extra_fields = dict(
        id=fields.StringField(label='id'),
    )
