from sqlalchemy import func
from urllib.parse import urlparse

from yaphone.newpdater.src.common import s3mds
from yaphone.newpdater.src.common.databases import updater_db as db


class Screenshot(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    s3key = db.Column(db.String, nullable=True)
    app_id = db.Column(db.Integer, db.ForeignKey('store_app.id'), nullable=False)

    @property
    def url(self):
        return s3mds.client.make_url(self.s3key)

    @property
    def preview_url(self):
        return self.url  # TODO: make resizer url in serializer SMARTTVBACKEND-9

    def __repr__(self):
        name = urlparse(self.preview_url).path
        return f'<Screenshot {name}>'


class StoreApp(db.Model):
    id = db.Column(db.Integer, primary_key=True)
    size = db.Column(db.Integer, nullable=False)
    app_name = db.Column(db.String, nullable=False)
    updated_at = db.Column(
        type_=db.DateTime(timezone=True),
        server_default=func.now(),
        onupdate=func.now(),
        nullable=False
    )
    _uploaded_at = db.Column(db.DateTime(timezone=True), name='uploaded_at', nullable=False)
    banner_s3key = db.Column(db.String, nullable=True)
    description = db.Column(db.String, nullable=False)
    package_name = db.Column(db.String(260), nullable=False)
    version_name = db.Column(db.String, nullable=False)
    version_code = db.Column(db.Integer, nullable=False)
    s3key = db.Column(db.String, nullable=True)
    installs_num = db.Column(db.Integer, default=0)
    company_name = db.Column(db.String, nullable=False)
    restriction_age = db.Column(db.Integer, nullable=False)
    eula_s3key = db.Column(db.String, nullable=True)
    privacy_s3key = db.Column(db.String, nullable=True)
    enabled = db.Column(db.Boolean, nullable=False, default=True)
    targeting_string = db.Column(db.String, nullable=True)

    category_id = db.Column(db.String, db.ForeignKey('category.id'), nullable=False)
    screenshots = db.relationship(Screenshot, backref='store_app', lazy='immediate', cascade='all, delete-orphan')

    def __repr__(self):
        return f'<StoreApp {self.id} {self.package_name}>'

    @property
    def url(self):
        return s3mds.client.make_url(self.s3key) if self.s3key else None

    @property
    def banner_url(self):
        return s3mds.client.make_url(self.banner_s3key) if self.banner_s3key else None

    @property
    def eula_url(self):
        return s3mds.client.make_url(self.eula_s3key) if self.eula_s3key else None

    @property
    def privacy_url(self):
        return s3mds.client.make_url(self.privacy_s3key) if self.privacy_s3key else None

    @property
    def uploaded_at(self):
        epoch_millis = self._uploaded_at.timestamp() * 1000
        return int(epoch_millis)


class Category(db.Model):
    id = db.Column(db.String, primary_key=True)
    name = db.Column(db.String, nullable=False)
