import logging
import uuid

from werkzeug.exceptions import NotFound

logger = logging.getLogger(__name__)


class StoreService:
    def __init__(self, loader, repository, targeting=None):
        self.loader = loader
        self.repository = repository
        self.targeting = targeting

    def filter(self, apps, request):
        filtered = []
        for app in apps:
            if app.targeting_string:
                # у приложения задан таргетинг, где его показывать
                logger.debug(f'targeting app {app.app_name}')
                if self.targeting.decide(app.targeting_string, request=request) == "hide":
                    logger.info(f'skipping app {app.app_name} as result of rule {app.targeting_string}')
                    continue
            filtered.append(app)

        return filtered

    def get_categories(self):
        return self.repository.find_categories()

    def get_apps(self, category='', request=None):
        if category:
            apps = self.repository.find_apps_by_category(category)
        else:
            apps = self.repository.find_apps()

        if self.targeting and request:  # not set in some tests
            apps = self.filter(apps, request=request)

        if apps:
            return apps
        else:
            raise NotFound(f'No apps found for category="{category}".')

    def get_app_url(self, package_name):
        url = self.repository.find_url_by_package(package_name)
        if url:
            return url
        else:
            error_message = f'No url found with package="{package_name}".'
            logger.exception(error_message)
            raise NotFound(error_message)

    def add_installation(self, package_name):
        edited = self.repository.add_installation(package_name)
        if not edited:
            error_message = f'No app found with package="{package_name}".'
            logger.exception(error_message)
            raise NotFound(error_message)

    @staticmethod
    def generate_app_s3key():
        return f'apps/{uuid.uuid4()}'

    @staticmethod
    def generate_img_s3key(package_name):
        return f'apps/{package_name}_imgs/{uuid.uuid4()}'

    @staticmethod
    def generate_doc_s3key(package_name):
        return f'doc/{package_name}/{uuid.uuid4()}'


class TargetingService:

    def __init__(self):
        self.mapping = {
            # строка в админке -> функция
            'only_module': self.target_only_module,
            'exclude_module': self.target_exclude_module,
            'not_rt2871_hikeen': self.target_exclude_2871,
            'not_rt2871_not_cvte6681': self.target_exclude_2871_6681,
        }

    def get_filter_function(self, key):
        return self.mapping.get(key)

    def decide(self, rule_string, **kwargs):
        """
        Применяет правило rule_string к переданным входным аргументам (например
        http заголовкам) и возвращает:
        - hide - не показывать приложение
        - show - показывать
        - None
        """
        fn = self.get_filter_function(rule_string)
        if fn:
            result = fn(**kwargs)
            logger.debug(f'{fn.__name__} returned {result} for {rule_string}')
            return result
        else:
            logger.warning(f'can not find filtering function for "{rule_string}"')

    @staticmethod
    def target_only_module(**kwargs):
        request = kwargs.get('request')
        if not request:
            logger.info('Can not apply targeting, because request is absent')
            return 'hide'

        platform = request.headers.get('X-YaQuasarPlatform')

        if platform == 'yandexmodule_2':
            return 'show'

        logger.debug(f'hide app because platform "{platform}" is not a module')
        return 'hide'

    @staticmethod
    def target_exclude_module(**kwargs):
        request = kwargs.get('request')
        if not request:
            logger.info('Can not apply targeting, because request is absent')
            return 'show'

        platform = request.headers.get('X-YaQuasarPlatform')

        if platform == 'yandexmodule_2':
            return 'hide'

        return 'show'

    @staticmethod
    def target_exclude_2871(**kwargs):
        request = kwargs.get('request')
        if not request:
            logger.info('Can not apply targeting, because request is absent')
            return 'show'

        platform = request.headers.get('X-YaQuasarPlatform')

        if platform == 'yandex_tv_rt2871_hikeen':
            return 'hide'

        return 'show'

    @staticmethod
    def target_exclude_2871_6681(**kwargs):
        request = kwargs.get('request')
        if not request:
            logger.info('Can not apply targeting, because request is absent')
            return 'show'

        platform = request.headers.get('X-YaQuasarPlatform')

        if platform in ('yandex_tv_rt2871_hikeen', 'yandex_tv_mt6681_cvte'):
            return 'hide'

        return 'show'
