import pytest
from werkzeug.exceptions import NotFound
from mock import Mock

from yaphone.newpdater.src.store import StoreService, TargetingService
from .fixtures import OKKO_APP, EXPLORER_APP, IVI_APP
from .mocks import StoreMockRepo

INCORRECT_CATEGORY = 'supergood_staff'
# We add only 'tv' and 'file_manager' categories
NON_ADDED_CATEGORY = 'browser'


def categories():
    return [
        {
            'id': 'cinema',
            'name': 'Кинотеатры'
        },
        {
            'id': 'tv',
            'name': 'TB'
        },
        {
            'id': 'file_manager',
            'name': 'Работа с файлами'
        },
        {
            'id': 'video_manager',
            'name': 'Работа с видео'
        },
        {
            'id': 'browser',
            'name': 'Браузеры'
        }
    ]


@pytest.fixture(name='categories')
def categories_fixture():
    return categories()


def all_apps():
    return [OKKO_APP, EXPLORER_APP, IVI_APP]


@pytest.fixture(name='all_apps')
def all_apps_fixture():
    return all_apps()


def service(categories, all_apps):
    return StoreService(repository=StoreMockRepo(
        apps=all_apps,
        categories=categories),
        loader=None
    )


@pytest.fixture(name='service')
def service_fixture(categories, all_apps):
    return service(categories, all_apps)


@pytest.fixture
def empty_service(categories):
    return StoreService(repository=StoreMockRepo(
        apps=[],
        categories=[]),
        loader=None
    )


def test_categories_should_return_all(service, categories):
    result = service.get_categories()
    assert result == categories


def test_categories_empty_result(empty_service):
    assert empty_service.get_categories() == []


def test_apps_returns_all_apps(service, all_apps):
    assert service.get_apps() == all_apps
    assert service.get_apps('') == all_apps
    assert service.get_apps(None) == all_apps


@pytest.mark.parametrize('app', all_apps())
def test_apps_returns_apps_for_category(service, app):
    assert service.get_apps(app.get('category')) == [app]


@pytest.mark.parametrize('category', [INCORRECT_CATEGORY, NON_ADDED_CATEGORY, 4, -1])
def test_apps_raises_not_found(service, category):
    with pytest.raises(NotFound) as context:
        service.get_apps(category)
    assert str(context.value) == f'404 Not Found: No apps found for category="{category}".'


class TestTargetingCases:
    @pytest.mark.parametrize('client_platform,expected', [
        ('yandexmodule_2', 'show'),
        ('hisi6681', 'hide'),
        (None, 'hide'),
    ])
    def test_only_module(self, client_platform, expected):
        """
        Таргетинг только для модуля
        """
        targeting = TargetingService()
        request = Mock()
        request.headers = {'X-YaQuasarPlatform': client_platform}
        assert targeting.decide('only_module', request=request) == expected

    def test_only_module_without_request(self):
        """
        Тест на устойчивость вызова без request
        """
        targeting = TargetingService()
        assert targeting.decide('only_module') == 'hide'


class TestFilterFunction:

    def build_store(self, apps):
        return StoreService(
            repository=StoreMockRepo(
                apps=apps,
                categories=[]
            ),
            loader=None,
            targeting=TargetingService(),
        )

    def test_only_module_filtered(self):
        """
        Приложение для модуля отфильтровывается из результатов не на модуле
        """
        okko = Mock(targeting_string='only_module')
        wink = Mock()
        kion = Mock()

        apps = [okko, wink, kion]
        service = self.build_store(apps)

        tv_request = Mock(headers={'X-YaQuasarPlatform': 'tv'})
        assert service.get_apps(request=tv_request) == [wink, kion]

        # и если платформа не указана, то тоже не покажется
        unknown_platform_request = Mock(headers={})
        assert service.get_apps(request=unknown_platform_request) == [wink, kion]

    def test_only_module_not_filtered_on_module(self):
        """
        Приложение только для модуля показывается на модуле
        """
        okko = Mock(targeting_string='only_module')
        wink = Mock()
        kion = Mock()

        apps = [okko, wink, kion]
        service = self.build_store(apps)

        tv_request = Mock(headers={'X-YaQuasarPlatform': 'yandexmodule_2'})
        assert service.get_apps(request=tv_request) == [okko, wink, kion]

    def test_not_rt2871_filtered(self):
        """
        Если стоит таргетинг not_rt2871_hikeen, то приложение не показывается
        на этой платформе
        """
        okko = Mock(targeting_string='not_rt2871_hikeen')
        wink = Mock()
        kion = Mock()

        apps = [okko, wink, kion]
        service = self.build_store(apps)

        # на 2871 окко не показывается
        tv_request = Mock(headers={'X-YaQuasarPlatform': 'yandex_tv_rt2871_hikeen'})
        assert service.get_apps(request=tv_request) == [wink, kion]

        # а на другой платформе показывается
        tv_request = Mock(headers={'X-YaQuasarPlatform': 'something'})
        assert service.get_apps(request=tv_request) == [okko, wink, kion]
