import logging

from yaphone.newpdater.src.common.databases import updater_db as db, on_error_retry
from yaphone.newpdater.src.common.exceptions import AppNotFound
from yaphone.newpdater.src.common.models import DEFAULT_LANGUAGE
from yaphone.newpdater.src.common.models import YandexDistributedApp
from yaphone.newpdater.src.common.serialization import serialize, serialize_many, YandexDistributedSerializer
from yaphone.newpdater.src.subscription.models import Subscription
from yaphone.newpdater.src.subscription.serializers import SubscriptionSerializer

logger = logging.getLogger(__name__)


# noinspection PyMethodMayBeStatic
class SubscriptionRepo:

    @on_error_retry
    def add_subscription(self, uuid, package_name):
        query = Subscription.query
        device = query.filter_by(device_id=uuid, package_name=package_name).first()
        if device:
            logger.error('%s with %s was already subscribed', uuid, package_name)
        else:
            device = Subscription(device_id=uuid, package_name=package_name)
            db.session.add(device)
            db.session.commit()

    @on_error_retry
    def remove_subscription(self, uuid, package_name):
        """
        Removes subscription, if possible.

        :return: `True` if subscription removed successfully
                 `False` otherwise
        """
        query = Subscription.query
        device = query.filter_by(device_id=uuid, package_name=package_name).first()
        if device:
            db.session.delete(device)
            db.session.commit()
            return True
        else:
            return False

    @on_error_retry
    def find_by_device_id(self, device_id):
        subscriptions = Subscription.query.filter_by(device_id=device_id).all()
        return serialize_many(SubscriptionSerializer, subscriptions)

    @on_error_retry
    def find_all(self):
        subscriptions = Subscription.query.all()
        return serialize_many(SubscriptionSerializer, subscriptions)

    @on_error_retry
    def find_distributed_app(self, package_name, lang=DEFAULT_LANGUAGE):
        """
        If this app can be installed bypassing Google Play, returns it
        """
        ya_distributed = YandexDistributedApp.objects(
            package_name__iexact=package_name
        ).first()
        if ya_distributed:
            serialized = serialize(YandexDistributedSerializer, ya_distributed)
            serialized['title'] = ya_distributed.get_title(lang)
            return serialized
        else:
            raise AppNotFound(package_name, 'yandex_distributed_apps')

    @on_error_retry
    def is_distributed_app(self, package_name):
        """
        Checks whether this app can be installed bypassing Google Play or not.
        """
        contains = YandexDistributedApp.objects(
            package_name__iexact=package_name
        ).count()
        return bool(contains)
