import json

import pytest
from flask import Flask

from yaphone.newpdater.src.core import settings, routes
from yaphone.newpdater.src.common import databases
from yaphone.newpdater.src.core.app import RegexpConverter
from yaphone.newpdater.src.common.models import YandexDistributedApp
from yaphone.newpdater.src.subscription.models import Subscription


def load_app_to_mongo():
    app_object = YandexDistributedApp.objects(package_name='com.yandex.launcher').first()
    if not app_object:
        app_object = YandexDistributedApp(package_name='com.yandex.launcher').save()
    return app_object


@pytest.fixture
def client(correct_data):
    app = Flask(__name__)
    app.config.from_object(settings.LocalTestConfig)
    app.url_map.converters['re'] = RegexpConverter
    routes.init_app(app)
    databases.init_app(app)
    app.app_context().push()
    databases.updater_db.create_all()

    subscription = Subscription(
        device_id=correct_data['device_id'],
        package_name=correct_data['package_name']
    )

    session = databases.updater_db.session
    session.add(subscription)
    session.commit()

    app_from_mongo = load_app_to_mongo()

    yield app.test_client()

    app_from_mongo.delete()
    session.remove()
    databases.updater_db.drop_all()


@pytest.fixture
def headers():
    return {
        'Content-Type': 'application/json'
    }


@pytest.fixture
def correct_data():
    return {
        'device_id': '048203cec91dd9c0d43a0bb332eb322c',
        'package_name': 'com.yandex.launcher'
    }


@pytest.fixture
def incorrect_data():
    return {
        'device_id': 'be7fa8e7bdcaf8e7adc9e7a0cea27e01',
        'package_name': 'ru.wasistdas.someapp'
    }


def test_subscribe_succeeds(client, headers, correct_data):
    url = '/api/v2/subscribe'

    response = client.post(url, data=json.dumps(correct_data), headers=headers)

    assert response.status_code == 200


def test_subscribe_raises_not_acceptable(client, headers, incorrect_data):
    url = '/api/v2/subscribe'

    response = client.post(url, data=json.dumps(incorrect_data), headers=headers)

    assert response.status_code == 406


def test_subscribe_and_unsubscribe_succeeds(client, headers, correct_data):
    client.post('/api/v2/subscribe', data=json.dumps(correct_data), headers=headers)
    response = client.post('/api/v2/unsubscribe', data=json.dumps(correct_data), headers=headers)

    assert response.status_code == 200


def test_unsubscribe_raises_not_found(client, headers, incorrect_data):
    url = '/api/v2/unsubscribe'

    response = client.post(url, data=json.dumps(incorrect_data), headers=headers)

    assert response.status_code == 404
