import pytest
from werkzeug.exceptions import NotAcceptable, NotFound

from yaphone.newpdater.src.mocks.subscriptions import SubscriptionMockRepo
from yaphone.newpdater.src.subscription import SubscriptionService

INCORRECT_UUID = 'be7fa8e7bdcaf8e7adc9e7a0cea27e01'
FIRST_CORRECT_UUID = '048203cec91dd9c0d43a0bb332eb322c'
SECOND_CORRECT_UUID = '1a01aa69c98e0f90a3cc8113fcd75dcd'


@pytest.fixture
def service():
    repo = SubscriptionMockRepo(
        distributed_apps=[
            {
                'device_id': FIRST_CORRECT_UUID,
                'package_name': 'com.yandex.launcher'
            },
            {
                'device_id': SECOND_CORRECT_UUID,
                'package_name': 'com.google.android.systemui'
            }
        ],
        subscriptions=[{
            'device_id': SECOND_CORRECT_UUID,
            'package_name': 'com.google.android.systemui'
        }]
    )
    service = SubscriptionService(repo)
    return service


def test_subscribe_succeeds(service):
    service.subscribe(FIRST_CORRECT_UUID, 'com.yandex.launcher')


def test_subscribe_raises_not_acceptable(service):
    with pytest.raises(NotAcceptable) as context:
        service.subscribe(INCORRECT_UUID, 'ru.wasistdas.someapp')

    assert str(context.value) == '406 Not Acceptable: App is not being watched'


def test_unsubscribe_succeeds(service):
    service.unsubscribe(SECOND_CORRECT_UUID, 'com.google.android.systemui')


def test_unsubscribe_raises_not_found(service):
    with pytest.raises(NotFound) as context:
        service.unsubscribe(INCORRECT_UUID, 'ru.wasistdas.someapp')

    assert str(context.value) == '404 Not Found: There is no such subscription.'


def test_subscribe_and_unsubscribe_succeeds(service):
    service.subscribe(FIRST_CORRECT_UUID, 'com.yandex.launcher')
    service.unsubscribe(FIRST_CORRECT_UUID, 'com.yandex.launcher')
