import logging
import tempfile

from flask_admin.form import rules
from markupsafe import Markup
from wtforms import fields

from yaphone.newpdater.src.common import validation
from yaphone.newpdater.src.admin.fields import ApkFileField, FirmwareFileField, ReadOnlyStringField
from yaphone.newpdater.src.admin.views.base import AdminModelView, DownloadUrlRowAction, FileUploadView
from yaphone.newpdater.src.updates import firmware_service, loader, models, updates_service

logger = logging.getLogger(__name__)


def description_formatter(view, context, model, name):
    value = getattr(model, name)
    if value is None:
        return None
    if len(value) > 350:
        value = f'{value[:350]}…'
    return Markup(value.replace('\n', '<br/>'))


def updates_formatter(view, context, model, name):
    updates = getattr(model, name)
    return Markup('<br/>'.join(repr(ota) for ota in updates))


class FirmwareChangelogsView(AdminModelView):
    form_extra_fields = dict(
        name=fields.StringField(label='Internal name'),
        title=fields.StringField(label='Title'),
        description=fields.TextAreaField(label='Description'),
    )

    column_searchable_list = (
        'title',
        'description',
    )

    column_labels = {
        'name': 'Internal name',
    }

    column_formatters = {
        'updates': updates_formatter,
        'description': description_formatter,
    }

    column_list = (
        'name',
        'title',
        'description',
        'updates',
    )

    form_columns = (
        'name',
        'title',
        'description',
        'updates',
    )

    form_widget_args = dict(
        description=dict(rows=10),
    )

    def validate_changelog(self, form):
        if form.title.data and not form.description.data:
            raise validation.ValidationError('Provided title without description')

    def on_model_change(self, form, model, is_created):
        self.validate_changelog(form)
        super().on_model_change(form, model, is_created)


def changelog_formatter(view, context, model, name):
    changelog = getattr(model, name)
    if changelog is None:
        return None

    description = changelog.description
    if len(description) > 300:
        description = f'{description[:300]}…'
    description = description.replace('\n', '<br/>')

    value = f"<i>{changelog.name}</i><br/>" \
            f"<b>{changelog.title or models.DEFAULT_CHANGELOG_TITLE}</b><br/>" \
            f"{description}"

    return Markup(value)


class FirmwareUpdatesView(FileUploadView):
    column_extra_row_actions = [
        DownloadUrlRowAction(lambda obj: obj.url),
    ]

    form_extra_fields = dict(
        overwrite=fields.BooleanField(default=False, label='Overwrite file'),
        file=FirmwareFileField(),
        url=ReadOnlyStringField(label='Download URL'),
        note=fields.TextAreaField(label='Note'),
    )

    column_searchable_list = ('os_version', 'build_version', 'manufacturer', 'product_name', 'model')

    column_list = (
        'enabled',
        'changelog',
        'branch',
        'os_version',
        'build_version',
        'manufacturer',
        'product_name',
        'model',
        'uploaded',
    )

    column_formatters = {
        'changelog': changelog_formatter,
    }

    form_widget_args = dict(
        filename=dict(disabled=True),
        url=dict(disabled=True),
        s3key=dict(disabled=True),
        md5_hash=dict(disabled=True),
        size=dict(disabled=True),
        note=dict(rows=5),
    )

    form_rules = [
        rules.Field('enabled'),
        rules.FieldSet((
            'os_version',
            'build_version',
            'manufacturer',
            'product_name',
            'model',
            'branch',
        ), 'Tagret'),
        rules.FieldSet((
            'uploaded',
            'filename',
            'size',
            's3key',
            'md5_hash',
            'url',
            'file',
            'overwrite',
        ), 'File'),
        rules.FieldSet((
            'changelog',
            'note',
        ), 'Info'),
    ]

    def on_model_change(self, form, model, is_created):
        super().on_model_change(form, model, is_created)
        filedata = self.validate_file_uploaded(form, is_created)
        if filedata:
            if is_created or model.s3key is None:
                model.s3key = firmware_service.generate_new_s3key()
            model.filename = filedata.filename
            overwrite = self.should_overwrite(form)
            loader.check_file_exists(model.s3key, overwrite, exception=validation.ValidationError)
            logger.debug(
                'on_model_change: %s can be loaded, overwriting: %s, loading...', model.s3key, overwrite
            )

    def after_model_change(self, form, model, is_created):
        filedata = self.validate_file_uploaded(form, is_created)
        if filedata:
            logger.debug('after_model_change: uploading %s to remote storage', model.s3key)
            temp_stream = tempfile.TemporaryFile()
            filedata.save(temp_stream, buffer_size=16*1024*1024)
            temp_stream.seek(0)
            firmware_service.upload_async(model.s3key, model.id, filestream=temp_stream, filename=filedata.filename)

    def after_model_delete(self, model):
        if model.s3key:
            firmware_service.delete_file(model.s3key)


class LocalUpdatesView(FileUploadView):
    column_extra_row_actions = [
        DownloadUrlRowAction(lambda obj: obj.download_url),
    ]

    form_extra_fields = dict(
        overwrite=fields.BooleanField(default=False, label='Overwrite file'),
        file=ApkFileField(),
        download_url=ReadOnlyStringField(),
        note=fields.TextAreaField(label='Note'),
    )

    column_searchable_list = ('app_name', 'package_name', 'branch')

    column_list = (
        'package_name',
        'is_system',
        'version_code',
        'version_name',
        'branch',
        'app_name',
        'size',
    )

    column_labels = {
        'app_name': 'Application name',
        'is_system': 'System package',
    }

    form_widget_args = dict(
        size=dict(disabled=True),
        filename=dict(disabled=True),
        download_url=dict(disabled=True),
        s3key=dict(disabled=True),
    )

    form_rules = [
        rules.FieldSet((
            'package_name',
            'branch',
            'version_name',
            'version_code',
            'app_name',
            'is_system',
        ), 'Package'),
        rules.FieldSet((
            'filename',
            'size',
            's3key',
            'download_url',
            'file',
            'overwrite',
        ), 'File'),
        rules.FieldSet((
            'note',
        ), 'Info'),
    ]

    def on_model_change(self, form, model, is_created):
        super().on_model_change(form, model, is_created)
        filedata = self.validate_file_uploaded(form, is_created)
        if filedata:
            if is_created or model.s3key is None:
                model.s3key = updates_service.generate_new_s3key()
            model.filename = filedata.filename
            overwrite = self.should_overwrite(form)
            loader.check_file_exists(model.s3key, overwrite, exception=validation.ValidationError)
            logger.debug(
                'on_model_change: %s can be loaded, overwriting: %s, loading...',
                model.s3key, overwrite
            )

            metadata = loader.upload_file(model.s3key, filedata.stream, filedata.filename)
            model.size = metadata.size

    def after_model_delete(self, model):
        if model.s3key:
            updates_service.delete_file(model.s3key)
