from typing import List, Optional

from yaphone.localization import UserSpecificConfig, Translator

from yaphone.newpdater.src.updates import models

TRACKED_BUILDS = 'tracked_builds'


class Prefix:
    TV = 'tv.'
    PHONE = 'phone.'
    LEGACY = 'legacy.'
    FOTA = 'fota.'


def get_branch_by_version(builds: list, version: str) -> Optional[str]:
    """
    Get a branch name from a list of builds.
    :param builds: e.g. [{'version': '7.1', 'branch': 'dev'}, {'version': '9.0.1', 'branch': 'tv'}]
    :param version: e.g. '7.1'
    :return: branch, e.g. 'dev', or None if there is no such branch.
    """
    branch = None
    for build in builds:
        if build.get('version') == version:
            branch = build.get('branch')
    return branch


def find_tracked_builds(config: UserSpecificConfig, prefix: str = '') -> list:
    """
    Finds tracked builds in localization.
    Attempts to load with a prefix. If fails, tries to load builds w/o prefix.
    :param prefix: user's prefix
    :param config: to load builds from
    :return: a list of builds, or an empty list []
    """
    builds = config.get_object(prefix + TRACKED_BUILDS, [])
    if isinstance(builds, list):
        return builds
    elif builds:
        return [builds]
    else:
        return []


def translate_updates(updates: List[models.LocalUpdate], translator: Translator) -> List[models.LocalUpdate]:
    for update in updates:
        key = update.app_name
        title = translator.translate(key) or key
        update.app_name = title
    return updates
