from sqlalchemy.orm import relationship

from yaphone.newpdater.src.common import s3mds
from yaphone.newpdater.src.common.databases import updater_db as db

DEFAULT_CHANGELOG_TITLE = 'Доступно новое обновление'


class AppUpdateMixin:
    id = db.Column(db.Integer, primary_key=True)
    package_name = db.Column(db.String(260), nullable=False)
    version_code = db.Column(db.Integer())
    version_name = db.Column(db.String())

    def __repr__(self):
        return '<%s %s:%s.%s>' % (self.__class__, self.package_name, self.version_code, self.version_name)


class S3UpdateMixin:
    branch = db.Column(db.String(), nullable=False)
    filename = db.Column(db.String(), nullable=True)
    s3key = db.Column(db.String(), nullable=True)


class LocalUpdate(AppUpdateMixin, S3UpdateMixin, db.Model):
    is_system = db.Column(db.Boolean(), default=False)
    size = db.Column(db.Integer())
    app_name = db.Column(db.String())

    note = db.Column(db.String(), nullable=True, default=None)

    @property
    def download_url(self):
        return s3mds.client.make_url(self.s3key)


class MarketUpdate(AppUpdateMixin, db.Model):
    title = db.Column(db.String())


class OtaUpdate(S3UpdateMixin, db.Model):
    id = db.Column(db.Integer, primary_key=True)
    enabled = db.Column(db.Boolean(), nullable=False, default=False)
    os_version = db.Column(db.String(), nullable=False)
    build_version = db.Column(db.String(), nullable=False)
    manufacturer = db.Column(db.String(), nullable=False)
    product_name = db.Column(db.String(), nullable=False)
    model = db.Column(db.String(), nullable=False)
    uploaded = db.Column(db.Boolean(), nullable=False, default=False)
    size = db.Column(db.Integer(), nullable=True, default=None)
    md5_hash = db.Column(db.String(32), nullable=True, default=None)

    changelog = relationship('Changelog', back_populates='updates')
    changelog_id = db.Column(db.Integer, db.ForeignKey('changelog.id'), nullable=True)

    note = db.Column(db.String(), nullable=True, default=None)

    def __repr__(self):
        return f'{self.manufacturer}, {self.model}, {self.os_version}, {self.build_version}'

    @property
    def url(self):
        return s3mds.client.make_url(self.s3key)


class Changelog(db.Model):
    max_html_len = 350

    id = db.Column(db.Integer, primary_key=True)
    name = db.Column(db.String(), nullable=True)
    title = db.Column(db.String(), nullable=True)
    description = db.Column(db.String(), nullable=False)

    updates = relationship('OtaUpdate', back_populates='changelog')

    def __eq__(self, other):
        try:
            return self.title == other.title and self.description == other.description
        except AttributeError:
            return False

    def __repr__(self):
        return f'{self.name} - {self.title or DEFAULT_CHANGELOG_TITLE}'
