from abc import ABC

from yaphone.newpdater.src.common.databases import on_error_retry
from yaphone.newpdater.src.common.serialization import deserialize


class UploadRepo(ABC):
    main_model = None
    main_deserializer = None
    filter_fields = None

    def __init__(self, session):
        self.session = session

    def add_and_get(self, data):
        self.add(data)
        self.commit()
        return self.find(data)

    @on_error_retry
    def add(self, data):
        update = deserialize(self.main_deserializer, data, session=self.session)
        self.session.add(update)
        self.session.flush()
        return update.id

    @on_error_retry
    def find(self, args):
        if self.filter_fields is not None:
            return self.session.query(
                self.main_model,
            ).filter_by(
                **self.filter_args(args)
            ).first()

    @on_error_retry
    def commit(self):
        self.session.commit()

    def filter_args(self, args) -> dict:
        """
        Removes arguments, which are not in a filter_fields.
        E.g.:
            filter_fields = ('a', 'b')

            args = {'a': 1, 'b': 2, 'c': 3, 'd': 4}
            filter_args(args)

            returns {'a': 1, 'b': 2}

        :param args: arguments to filter
        :return: filtered dict
        """
        return {field: args[field] for field in self.filter_fields}
