from yaphone.utils.parsed_version import ParsedVersion

from yaphone.newpdater.src.common.databases import on_error_retry
from yaphone.newpdater.src.common.exceptions import UpdateNotFound
from yaphone.newpdater.src.updates.models import OtaUpdate, Changelog
from yaphone.newpdater.src.updates.repository.base import UploadRepo
from yaphone.newpdater.src.updates.serializers import OtaUpdateDeserializer


class FirmwareRepo(UploadRepo):
    main_model = OtaUpdate
    main_deserializer = OtaUpdateDeserializer
    filter_fields = ('os_version', 'build_version', 'manufacturer', 'model', 'branch',)

    @on_error_retry
    def add_changelog(self, update_id, title, description):
        changelog = Changelog(title=title, description=description)
        self.session.add(changelog)  # TODO: try to merge instead
        changelog = self.find_changelog_id(title, description)
        self.session.query(
            OtaUpdate
        ).filter_by(
            id=update_id,
        ).update(
            dict(changelog_id=changelog.id)
        )

    @on_error_retry
    def update_changelog(self, changelog_id, title, description):
        self.session.query(
            Changelog
        ).filter_by(
            id=changelog_id,
        ).update(
            dict(title=title, description=description),
            synchronize_session='fetch',
        )

    @on_error_retry
    def update_filename(self, update_id, filename):
        self.session.query(
            OtaUpdate,
        ).filter_by(
            id=update_id
        ).update(
            dict(filename=filename),
            synchronize_session='fetch',
        )

    @on_error_retry
    def update_note(self, update_id, note):
        self.session.query(
            OtaUpdate,
        ).filter_by(
            id=update_id
        ).update(
            dict(note=note),
            synchronize_session='fetch',
        )

    @on_error_retry
    def update_metadata(self, update_id, metadata):
        self.session.query(
            OtaUpdate,
        ).filter_by(
            id=update_id
        ).update(
            dict(
                size=metadata.size,
                md5_hash=metadata.md5_hash,
                uploaded=True,
            ),
            synchronize_session='fetch',
        )

    @on_error_retry
    def clear_uploaded(self, update_id):
        self.session.query(
            OtaUpdate,
        ).filter_by(
            id=update_id
        ).update(
            dict(
                uploaded=False,
            ),
            synchronize_session='fetch',
        )

    @on_error_retry
    def find_changelog_id(self, title, description):
        return self.session.query(
            Changelog.id,
        ).filter_by(
            title=title,
            description=description,
        ).order_by(
            Changelog.id.desc()
        ).first()

    @on_error_retry
    def find_ota_update_id(self, data):
        return self.session.query(
            OtaUpdate.id,
        ).filter_by(
            **data
        ).first()

    @on_error_retry
    def delete_update_by_id(self, update_id):
        self.session.query(
            OtaUpdate.id
        ).filter_by(
            id=update_id
        ).delete()

    @on_error_retry
    def find_ota_update(self, os_version, branch, brand, product_name, model_name):
        updates = OtaUpdate.query.filter(
            OtaUpdate.manufacturer.ilike(brand),
            OtaUpdate.product_name.ilike(product_name),
            OtaUpdate.model.ilike(model_name),
        ).filter_by(
            os_version=os_version,
            branch=branch,
            uploaded=True,
            enabled=True,
        ).outerjoin(
            Changelog
        ).all()

        if updates:
            # key function is called exactly once for each element
            return max(updates, key=lambda update: ParsedVersion(update.build_version))
        else:
            raise UpdateNotFound(f'{os_version}, {branch}', 'ota updates')

    @on_error_retry
    def find_all_ota_updates(self):
        updates = OtaUpdate.query.filter_by(
            uploaded=True
        ).all()
        # key function is called exactly once for each element
        sorted_updates = sorted(updates, key=lambda update: ParsedVersion(update.build_version), reverse=True)
        return sorted_updates
