import logging

from yaphone.newpdater.src.common.databases import updater_db as db, on_error_retry
from yaphone.newpdater.src.common.exceptions import UpdateNotFound
from yaphone.newpdater.src.updates.models import MarketUpdate, LocalUpdate, AppUpdateMixin
from yaphone.newpdater.src.updates.repository.base import UploadRepo
from yaphone.newpdater.src.updates.serializers import LocalUpdateDeserializer
from yaphone.newpdater.src.updates.types import PriorityEnum

logger = logging.getLogger(__name__)


class LocalUpdateRepo(UploadRepo):
    main_model = LocalUpdate
    main_deserializer = LocalUpdateDeserializer
    filter_fields = ('package_name', 'version_code', 'version_name', 'branch',)

    @on_error_retry
    def find_market_updates(self, filter_packages=None):
        return self._find_app_updates(filter_packages, MarketUpdate)

    @on_error_retry
    def find_local_updates(self, packages=None):
        return self._find_app_updates(packages, LocalUpdate)

    def _find_app_updates(self, packages, model=AppUpdateMixin):
        filter_by = db.session.query(model).filter_by

        existing_updates = []
        for package in packages:
            logger.debug('Find updates for package: %s', package)
            filters = {key: value for key, value in package.items() if key in ('package_name', 'branch')}
            update = filter_by(
                **filters
            ).order_by(
                model.version_code.desc()
            ).first()
            if update:
                update.priority = package.get('priority', PriorityEnum.NORMAL)
                existing_updates.append(update)
                logger.debug('Found an update: %s', update)
            else:
                logger.debug('Update not found')

        if existing_updates:
            return existing_updates
        else:
            raise UpdateNotFound(packages, 'updates')

    @on_error_retry
    def update(self, update_id, size, filename):
        return self.session.query(
            self.main_model,
        ).filter_by(
            id=update_id
        ).update(
            dict(size=size, filename=filename),
            synchronize_session='fetch',
        )
