from marshmallow import fields
from marshmallow_sqlalchemy import ModelSchema

from yaphone.newpdater.src.common.serialization import NullRemoverMixin
from yaphone.newpdater.src.updates import models, types


class OtaUpdateDeserializer(ModelSchema):
    uploaded = fields.Boolean(required=False, missing=False)
    enabled = fields.Boolean(required=False, missing=False)

    class Meta:
        model = models.OtaUpdate
        exclude = ('id',)


class ChangelogDeserializer(ModelSchema):
    class Meta:
        model = models.Changelog
        exclude = ('id',)


class LocalUpdateDeserializer(ModelSchema):
    class Meta:
        model = models.LocalUpdate
        exclude = ('id',)


class MarketUpdateSerializer(ModelSchema):
    class Meta:
        model = models.MarketUpdate
        fields = (
            'title',
            'package_name',
            'version_code',
            'version_name',
        )


class LocalUpdateSerializer(ModelSchema):
    class Meta:
        model = models.LocalUpdate
        fields = (
            'app_name',
            'size',
            'package_name',
            'version_code',
            'version_name',
            'download_url',
        )


class LocalUpdateV3Serializer(ModelSchema):
    bit_settings = fields.Method(serialize='serialize_bit_settings')
    priority = fields.String(default=types.PriorityEnum.NORMAL.value)

    @staticmethod
    def serialize_bit_settings(obj):
        value = 0b1 if obj.is_system else 0b0
        return value

    class Meta:
        model = models.LocalUpdate
        fields = (
            'app_name',
            'size',
            'package_name',
            'version_code',
            'version_name',
            'download_url',
            'bit_settings',
            'priority',
        )


class UpdatesV3Serializer(ModelSchema):
    updates = fields.Nested(LocalUpdateV3Serializer, many=True)


class ChangelogSerializer(ModelSchema):
    title = fields.Function(lambda obj: obj.title or models.DEFAULT_CHANGELOG_TITLE)

    class Meta:
        model = models.Changelog
        fields = (
            'title',
            'description',
        )


class OtaSerializer(NullRemoverMixin, ModelSchema):
    version = fields.String(attribute='build_version')
    changelog = fields.Nested(ChangelogSerializer)

    class Meta:
        model = models.OtaUpdate
        fields = (
            'version',
            'url',
            'changelog',
        )


class OtaV3Serializer(OtaSerializer):
    size = fields.String(attribute='size')
    md5_hash = fields.String(attribute='md5_hash')
    priority = fields.String(default=types.PriorityEnum.NORMAL.value)

    class Meta:
        model = models.OtaUpdate
        fields = (
            'version',
            'url',
            'changelog',
            'size',
            'md5_hash',
            'priority',
        )


class OtaSerializerNoChangelog(OtaSerializer):
    class Meta:
        model = models.OtaUpdate
        fields = (
            'version',
            'url',
        )
