import logging
import uuid
from datetime import datetime
from typing import Optional

from flask import copy_current_request_context
from threading import Thread
from werkzeug.exceptions import BadRequest

from yaphone.newpdater.src.common.exceptions import AppNotFound, UpdateNotFound
from yaphone.newpdater.src.common.validation import validate_many
from yaphone.newpdater.src.updates import localization, models, validators

logger = logging.getLogger(__name__)


def datetime_from(milliseconds):
    return datetime.fromtimestamp(int(milliseconds / 1000))


class LocalUpdateService:
    def __init__(self, loader, repository, subscriptions_repo):
        self.loader = loader
        self.repository = repository
        self.subscriptions_repo = subscriptions_repo

    def _load_subscriptions_data(self, device_id, locale):
        titles = {}
        packages = []
        for sub in self.subscriptions_repo.find_by_device_id(device_id):
            package = sub['package_name']
            try:
                app = self.subscriptions_repo.find_distributed_app(package, locale)
                titles.update({package: app['title']})
                packages.append(package)
            except AppNotFound as e:
                logger.info(e)
        return titles, packages

    def _change_titles(self, titles, updates):
        if not titles:
            return updates

        for update in updates:
            title = titles[update.package_name]
            if title:
                update.title = title

        return updates

    def get_market_updates(self, device_id, locale):
        try:
            titles, packages = self._load_subscriptions_data(device_id, locale)
            updates = self.repository.find_market_updates(packages)
            return self._change_titles(titles, updates)
        except UpdateNotFound:
            logger.info('No updates were found for %s', device_id)
            return []

    def get_local_updates_legacy(self, config, translator):
        return self.get_local_updates(localization.Prefix.LEGACY, config, translator)

    def get_local_updates_for_device(self, device_type, config, translator):
        prefix = f'{device_type.lower()}.' if device_type else ''
        return self.get_local_updates(prefix, config, translator)

    def get_local_updates(self, prefix, config, translator):
        try:
            builds = localization.find_tracked_builds(config, prefix)
            builds = validate_many(validators.TrackedBuildValidator, builds)
            updates = self.repository.find_local_updates(builds)
            return localization.translate_updates(updates, translator)
        except UpdateNotFound:
            logger.info('No updates were found for "%s"', prefix)
            return []
        except BadRequest:
            logger.info('No tracked builds for "%s"', prefix)
            return []

    @staticmethod
    def generate_new_s3key():
        return f'updates/{uuid.uuid4()}'

    def save(self, data):
        model = self.repository.find(data)
        if not model:
            data['s3key'] = self.generate_new_s3key()
            self.repository.add(data)
            model = self.repository.find(data)
        self.loader.check_file_exists(model.s3key, overwrite=data.get('overwrite', False))
        metadata = self.loader.upload_file(model.s3key, data['stream'], data['filename'])
        self.repository.update(model.id, metadata.size, data['filename'])
        self.repository.commit()

    def delete_file(self, s3key):
        if s3key:
            self.loader.delete_file(s3key)


class FirmwareService:
    def __init__(self, loader, repository):
        self.loader = loader
        self.repository = repository

    def get_ota_update(self, config, product) -> Optional[models.OtaUpdate]:
        try:
            builds = localization.find_tracked_builds(config, localization.Prefix.FOTA)
            validated_builds = validate_many(validators.TrackedOtaValidator, builds)
            branch = localization.get_branch_by_version(validated_builds, product.version)
            update = self.repository.find_ota_update(
                os_version=product.version,
                branch=branch,
                brand=product.brand,
                product_name=product.name,
                model_name=product.device
            )
            return update
        except UpdateNotFound:
            logger.info('No ota update found for %s and "%s"', product, localization.Prefix.FOTA)
            return None
        except BadRequest:
            msg = 'Tracked builds config is incorrect or empty for %s and "%s"'
            logger.warning(msg, product, localization.Prefix.FOTA)
            return None

    def get_ota_updates(self):
        return self.repository.find_all_ota_updates()

    def upload_async(self, key, update_id, filestream, filename):
        @copy_current_request_context
        def _upload_update(on_load, on_error):
            try:
                logger.debug('Start to upload %s to %s', filename, key)
                metadata = self.loader.upload_file(key, filestream, filename)
                logger.debug('Uploaded %s to %s', filename, key)
                on_load(metadata)
            except Exception as e:
                on_error(e)

        def _on_load(metadata):
            self.repository.update_metadata(update_id, metadata)
            self.repository.commit()
            logger.info('Update %s saved with id: %s', filename, update_id)

        def _on_error(exception):
            self.repository.clear_uploaded(update_id)
            self.repository.commit()
            logger.error('Loading of update %s aborted', filename)
            raise exception

        thread = Thread(target=_upload_update, args=(_on_load, _on_error))
        thread.start()

    @staticmethod
    def generate_new_s3key():
        return f'firmware/{uuid.uuid4()}'

    def save(self, data):
        model = self.repository.find(data)
        if not model:
            data['s3key'] = self.generate_new_s3key()
            model = self.repository.add_and_get(data)
        else:
            self.repository.update_filename(model.id, data['filename'])
            if 'note' in data:
                self.repository.update_note(model.id, data['note'])
            self.repository.commit()
        self.loader.check_file_exists(model.s3key, data.get('overwrite', False))
        self.upload_async(model.s3key, model.id, data['stream'], data['filename'])

    def save_all(self, data, changelog):
        model = self.repository.find(data)

        if not model:
            data['s3key'] = self.generate_new_s3key()
            model = self.repository.add_and_get(data)
            self.repository.add_changelog(model.id, **changelog)
        else:
            if model.changelog_id:
                self.repository.update_changelog(model.changelog_id, **changelog)
            else:
                self.repository.add_changelog(model.id, **changelog)
            self.repository.update_filename(model.id, data['filename'])
            if 'note' in data:
                self.repository.update_note(model.id, data['note'])
        self.repository.commit()

        self.loader.check_file_exists(model.s3key, data.get('overwrite', False))
        self.upload_async(model.s3key, model.id, data['stream'], data['filename'])

    def delete_file(self, s3key):
        if s3key:
            self.loader.delete_file(s3key)
