import pytest

from .conftest import put_to_bucket, get_filedata
from .fixtures import (
    firmware_args_no_changelog, firmware_data_no_changelog,
    different_args_no_changelog,
)
from yaphone.newpdater.src.common.databases import updater_db
from yaphone.newpdater.src.updates import firmware_service
from yaphone.newpdater.src.updates.repository.firmware import FirmwareRepo


class TestFirmwareUpload:
    url = '/int-api/v2/ota_upload'
    repo = FirmwareRepo(updater_db.session)

    @staticmethod
    def assert_required_args_matches(args, update):
        assert update
        assert args['os_version'] == update.os_version
        assert args['build_version'] == update.build_version
        assert args['manufacturer'] == update.manufacturer
        assert args['model'] == update.model

    @staticmethod
    def assert_changelog_matches(args, changelog):
        assert changelog
        assert args['title'] == changelog.title
        assert args['description'] == changelog.description

    @pytest.mark.parametrize('args', [firmware_args_no_changelog()])
    def test_add_new_update_no_changelog_succeed(self, client, args):
        args['file'] = get_filedata(filename='NotAFirmware2.zip')

        response = client.post(self.url, data=args)
        update = self.repo.find(args)

        assert response.status_code == 200
        assert not update.changelog
        self.assert_required_args_matches(args, update)

    @pytest.mark.parametrize(
        'args, data',
        [(firmware_args_no_changelog(), firmware_data_no_changelog())]
    )
    def test_overwrite_duplicate_update_succeeds(self, client, args, data):
        data['s3key'] = firmware_service.generate_new_s3key()
        self.repo.add(data=data)

        args['file'] = get_filedata(filename='NotAFirmware.zip')
        args['overwrite'] = True

        put_to_bucket(args['file'], data['s3key'])

        response = client.post(self.url, data=args)
        update = self.repo.find(args)

        assert response.status_code == 200
        self.assert_required_args_matches(args, update)

    @pytest.mark.parametrize(
        'args, data',
        [(firmware_args_no_changelog(), firmware_data_no_changelog())]
    )
    def test_add_duplicate_update_raises_bad_request(self, client, args, data):
        data['s3key'] = firmware_service.generate_new_s3key()
        self.repo.add(data=data)

        args['file'] = get_filedata(filename='NotAFirmware.zip')
        put_to_bucket(args['file'], data['s3key'])

        response = client.post(self.url, data=args)

        assert response.status_code == 400

    @pytest.mark.parametrize('args', different_args_no_changelog())
    @pytest.mark.parametrize('data', [firmware_data_no_changelog()])
    def test_add_another_update_overwrite_file_succeeds(self, client, args, data):
        data['s3key'] = firmware_service.generate_new_s3key()
        self.repo.add(data=data)
        args['file'] = get_filedata(filename='NotAFirmware.zip')
        args['overwrite'] = True

        response = client.post(self.url, data=args)
        update = self.repo.find(args)

        assert response.status_code == 200
        self.assert_required_args_matches(args, update)

    @pytest.mark.parametrize(
        'args, data',
        [(firmware_args_no_changelog(build_version='252'), firmware_data_no_changelog())]
    )
    def test_add_another_update_another_file_succeeds(self, client, args, data):
        data['s3key'] = firmware_service.generate_new_s3key()
        self.repo.add(data=data)
        args['file'] = get_filedata(filename='NotAFirmware4.zip')
        args['overwrite'] = False

        response = client.post(self.url, data=args)
        update = self.repo.find(args)

        assert response.status_code == 200
        self.assert_required_args_matches(args, update)
