import pytest

from .conftest import put_to_bucket, get_filedata
from .fixtures import app_args, app_data
from yaphone.newpdater.src.common.databases import updater_db
from yaphone.newpdater.src.common.s3mds import s3key
from yaphone.newpdater.src.updates.repository.local_updates import LocalUpdateRepo


class AppTestKey(s3key.DictKey):
    key_format = 'updates/{branch}/{filename}'
    # TODO: uncomment after starting to use tagged keys from updates/keys.py
    # key_format = 'updates/{branch}/{tag}/{filename}'
    # tag_fields = ('app_name', 'version_name', 'version_code', 'package_name', 'branch', 'filename')


class TestLocalUpload:
    url = '/int-api/v2/upload'
    repo = LocalUpdateRepo(updater_db.session)

    @staticmethod
    def assert_required_args_matches(args, update):
        assert update
        assert args['app_name'] == update.app_name
        assert args['version_name'] == update.version_name
        assert args['version_code'] == update.version_code
        assert args['package_name'] == update.package_name

    @pytest.mark.parametrize('args', [app_args()])
    def test_add_new_update_succeed(self, client, args):
        args['file'] = get_filedata('NotAnApk.apk')

        response = client.post(self.url, data=args)
        update = self.repo.find(args)

        assert response.status_code == 200
        self.assert_required_args_matches(args, update)

    @pytest.mark.parametrize('args, data', [(app_args(), app_data(filename='NotAnApk2.apk'))])
    def test_overwrite_duplicate_update_succeeds(self, client, args, data):
        self.repo.add(data=data)

        args['file'] = get_filedata('NotAnApk2.apk')
        args['overwrite'] = True
        key = AppTestKey(data).key
        put_to_bucket(args['file'], key)

        response = client.post(self.url, data=args)
        update = self.repo.find(args)

        assert response.status_code == 200
        self.assert_required_args_matches(args, update)

    @pytest.mark.parametrize('args, data', [(app_args(), app_data(filename='NotAnApk3.apk'))])
    def test_add_duplicate_update_raises_bad_request(self, client, args, data):
        self.repo.add(data=data)

        args['file'] = get_filedata('NotAnApk3.apk')
        key = AppTestKey(data).key
        put_to_bucket(args['file'], key)

        response = client.post(self.url, data=args)
        assert response.status_code == 400

    @pytest.mark.parametrize('args, data', [(app_args(), app_data(filename='NotAFirmware.zip'))])
    def test_incorrect_filetype_raises_bad_request(self, client, args, data):
        args['file'] = get_filedata('NotAFirmware.zip')

        response = client.post(self.url, data=args)

        assert response.status_code == 400
