import pytest

from yaphone.newpdater.src.mocks.subscriptions import SubscriptionMockRepo
from yaphone.newpdater.src.common import s3mds, serialization
from yaphone.newpdater.src.updates import serializers
from yaphone.newpdater.src.updates.service import LocalUpdateService, FirmwareService
from yaphone.newpdater.src.updates.localization import TRACKED_BUILDS, Prefix
from .mocks import UpdatesMockRepo
from .fixtures import (
    PHONE_TRACKED_BUILDS, TV_TRACKED_BUILDS, NO_PREFIX_TRACKED_BUILDS,
    OREO_DEV_OTA_TRACKED_BUILD, PIE_TV_DEV_OTA_UPDATE, EXISTING_DEVICE_OREO,
    super_app_local_update, ignored_app_local_update, auto_ru_local_update,
    home_app_local_update, ivi_local_update,
    OREO_DEV_OTA_UPDATE, OREO_PROD_OTA_UPDATE, NON_EXISTENT_DEVICE,
    EXISTING_DEVICE_PIE, EXISTING_RANDOM_CASE_DEVICE_PIE,
    OREO_TEST_OTA_TRACKED_BUILD, MM_DEV_OTA_TRACKED_BUILD,
)
from .mocks import UserConfigMock, MockTranslator

CORRECT_UUID = '048203cec91dd9c0d43a0bb332eb322c'
INCORRECT_UUID = 'be7fa8e7bdcaf8e7adc9e7a0cea27e01'

APP_NAMES = [
    'com.yandex.launcher.updaterapp',
    'com.yandex.launcher'
]
# device_type, expected_build
LOCAL_UPDATES_FOR_DEVICES = [
    ('legacy', [
        auto_ru_local_update(download_url='test://updates/dev/autoru.apk'),
        super_app_local_update(download_url='test://updates/dev/super_app.apk')
    ]),
    ('phone', [
        auto_ru_local_update(download_url='test://updates/dev/autoru.apk'),
        super_app_local_update(download_url='test://updates/dev/super_app.apk')
    ]),
    ('tv', [
        home_app_local_update(download_url='test://updates/tv_dev/home_app.apk'),
        ivi_local_update(download_url='test://updates/tv_dev/ivi.apk')
    ])
]
MOST_COMMON_APPS = [
    auto_ru_local_update(download_url='test://updates/dev/autoru.apk'),
    super_app_local_update(download_url='test://updates/dev/super_app.apk'),
]
UPDATES_FOR_DEVICES = [
    ('legacy', MOST_COMMON_APPS),
    ('phone', MOST_COMMON_APPS),
    ('tv', [
        home_app_local_update(download_url='test://updates/tv_dev/home_app.apk'),
        ivi_local_update(download_url='test://updates/tv_dev/ivi.apk')
    ])
]
# device, result_update
OTA_UPDATES_FOR_DEVICES = [
    (EXISTING_DEVICE_OREO, OREO_DEV_OTA_UPDATE),
    (EXISTING_DEVICE_PIE, PIE_TV_DEV_OTA_UPDATE),
    (EXISTING_RANDOM_CASE_DEVICE_PIE, PIE_TV_DEV_OTA_UPDATE)
]

OREO_TEST_CONFIG = UserConfigMock({
    Prefix.FOTA + TRACKED_BUILDS: OREO_TEST_OTA_TRACKED_BUILD,
})

MM_DEV_CONFIG = UserConfigMock({
    Prefix.FOTA + TRACKED_BUILDS: MM_DEV_OTA_TRACKED_BUILD,
})

DISTRIBUTED_APPS = [
    {
        'package_name': 'com.yandex.launcher',
        'title': 'wyrzutnia'
    },
    {
        'package_name': 'com.google.android.systemui',
        'title': 'interfejs systemu'
    }
]


@pytest.fixture
def patch_s3mds_client():
    url_template = s3mds.client.url_template
    s3mds.client.url_template = 'test://{key}'
    yield
    s3mds.client.url_template = url_template


@pytest.fixture
def config():
    return UserConfigMock({
        TRACKED_BUILDS: NO_PREFIX_TRACKED_BUILDS,
        Prefix.LEGACY + TRACKED_BUILDS: PHONE_TRACKED_BUILDS,
        Prefix.PHONE + TRACKED_BUILDS: PHONE_TRACKED_BUILDS,
        Prefix.TV + TRACKED_BUILDS: TV_TRACKED_BUILDS,
        Prefix.FOTA + TRACKED_BUILDS: OREO_DEV_OTA_TRACKED_BUILD,
    })


@pytest.fixture
def market_updates():
    return [
        {
            'package_name': 'com.google.android.systemui',
            'version_code': 1,
            'version_name': '1.1-24',
            'title': 'SystemUI'
        },
        {
            'package_name': 'com.rocket.launcher',
            'version_code': 2,
            'version_name': '2.1',
            'title': 'SuperLauncher'
        },
        {
            'package_name': 'com.yandex.autoru',
            'version_code': 4,
            'version_name': '4.3.3',
            'title': 'AutoRu'
        },
    ]


@pytest.fixture
def local_updates():
    return [
        auto_ru_local_update(filename='autoru.apk', branch='dev', s3key='updates/dev/autoru.apk'),
        ignored_app_local_update(),
        home_app_local_update(filename='home_app.apk', branch='tv_dev', s3key='updates/tv_dev/home_app.apk'),
        ivi_local_update(filename='ivi.apk', branch='tv_dev', s3key='updates/tv_dev/ivi.apk'),
        super_app_local_update(app_name='super_app', filename='super_app.apk', branch='dev', s3key='updates/dev/super_app.apk')
    ]


@pytest.fixture
def ota_updates():
    return [
        PIE_TV_DEV_OTA_UPDATE,
        OREO_DEV_OTA_UPDATE,
        OREO_PROD_OTA_UPDATE,
    ]


@pytest.fixture
def updates_repo(market_updates, local_updates, ota_updates):
    return UpdatesMockRepo(
        distributed_apps=DISTRIBUTED_APPS,
        local_updates=local_updates,
        market_updates=market_updates,
        ota_updates=ota_updates
    )


@pytest.fixture
def service(updates_repo):
    subscriptions = [{
        'device_id': CORRECT_UUID,
        'package_name': 'com.google.android.systemui'
    }]
    subscription_repo = SubscriptionMockRepo(
        distributed_apps=DISTRIBUTED_APPS,
        subscriptions=subscriptions
    )
    return LocalUpdateService(
        loader=None,
        repository=updates_repo,
        subscriptions_repo=subscription_repo,
    )


@pytest.fixture
def ota_service(updates_repo):
    return FirmwareService(
        loader=None,
        repository=updates_repo,
    )


@pytest.fixture
def translator():
    return MockTranslator({'super_app': 'CYBER APP'})


@pytest.mark.parametrize('device_id, locale', [(CORRECT_UUID, 'ru_RU')])
def test_market_updates_found(device_id, locale, service):
    expected_market_update = [{
        'package_name': 'com.google.android.systemui',
        'version_code': 1,
        'version_name': '1.1-24',
        'title': 'interfejs systemu'
    }]
    updates = service.get_market_updates(device_id, locale)
    result = serialization.serialize_many(serializers.MarketUpdateSerializer, updates)
    assert result == expected_market_update


@pytest.mark.parametrize('device_id, locale', [(INCORRECT_UUID, 'ru_RU')])
def test_market_updates_not_found_shows_empty_list(device_id, locale, service):
    result = service.get_market_updates(device_id, locale)
    assert result == []


@pytest.mark.parametrize('app_name', APP_NAMES)
def test_local_updates_for_app_found(app_name, config, service, translator, patch_s3mds_client):
    updates = service.get_local_updates_legacy(config, translator)
    result = serialization.serialize_many(serializers.LocalUpdateSerializer, updates)
    assert result == MOST_COMMON_APPS


@pytest.mark.parametrize('device_type, expected_build', LOCAL_UPDATES_FOR_DEVICES)
def test_local_updates_for_device_found(device_type, expected_build, config, service, translator, patch_s3mds_client):
    updates = service.get_local_updates_for_device(device_type, config, translator)
    result = serialization.serialize_many(serializers.LocalUpdateSerializer, updates)
    assert result == expected_build


class TestOtaUpdate:

    @pytest.mark.parametrize('device, result_update', OTA_UPDATES_FOR_DEVICES)
    def test_update_for_existing_device_found(self, ota_service, config, device, result_update, patch_s3mds_client):
        update = ota_service.get_ota_update(config, device)
        assert update.branch == result_update['branch']
        assert update.filename == result_update['filename']
        assert update.build_version == result_update['build_version']

    @pytest.mark.parametrize('device', [EXISTING_DEVICE_OREO])
    @pytest.mark.parametrize('config', [OREO_TEST_CONFIG, MM_DEV_CONFIG])
    def test_update_for_existing_device_not_found(self, ota_service, config, device):
        update = ota_service.get_ota_update(config, device)
        assert update is None

    @pytest.mark.parametrize('device', [NON_EXISTENT_DEVICE])
    def test_ota_update_no_such_device_not_found(self, ota_service, config, device, patch_s3mds_client):
        update = ota_service.get_ota_update(config, device)
        assert update is None
