import tempfile

from marshmallow import Schema, pre_load, ValidationError, validates, validates_schema
from marshmallow.fields import List
from marshmallow.fields import Raw, String, Integer, Boolean
from marshmallow.validate import OneOf, Regexp
from marshmallow.validate import Range
from yaphone.utils.parsers import parsers

from yaphone.newpdater.src.common.validation import HeadersValidator, BUILD_FIELD
from yaphone.newpdater.src.updates.types import UpdateType

Type = String(validate=OneOf(UpdateType.TYPES))


class ArgsList(List):

    def __init__(self, cls_or_instance, load_from, **kwargs):
        self.name = load_from
        super().__init__(cls_or_instance, **kwargs)

    def _deserialize(self, value, attr, obj, **kwargs):
        if hasattr(obj, 'getlist'):
            return obj.getlist(self.name)
        else:
            raise self.fail('Invalid object type %s: should contain getlist()' % type(obj))


class UpdateValidator(Schema):
    device_id = String(required=False)
    locale = String(required=False, validate=Regexp(parsers.ACCEPT_LANGUAGE_REGEXP))
    type = ArgsList(Type, load_from='type', required=False, default=UpdateType.DEFAULT)

    @validates_schema
    def validate_device_id(self, data):
        types = data.get('type')
        uuid = data.get('device_id')
        if has_google_play(types) and not uuid:
            raise ValidationError('device_id should not be empty for type google_play!')


def validate_language(accept_language, locale_param):
    """TODO: remove locale parameter and use only accept_language after changing on a client side"""
    if accept_language:
        locale = parsers.AcceptLanguageParser.parse(accept_language)
        if not locale:
            raise ValidationError('"Accept-Language" header has invalid format')
    else:
        if not locale_param:
            raise ValidationError('"Accept-Language" header or "locale" parameter should not be empty')
        locale = parsers.AcceptLanguageParser.parse(locale_param)
        if not locale:
            raise ValidationError('"locale" parameter has invalid format')
    return locale['language']


def has_google_play(types):
    if types:
        return UpdateType.GOOGLE_PLAY in types
    else:
        return False


class TrackedBuildValidator(Schema):
    """
    Validate tracked_builds from the localization-admin.
    Indented to use with validate_many() as we assume that
    there is a list of dicts with tracked builds in a format:
        [
          {
            "package_name": "com.yandex.launcher",
            "branch": "dev"
          }
        ]
    """
    package_name = String(required=True)
    branch = String(required=False)
    priority = String(required=False)


class TrackedOtaValidator(Schema):
    """
    Validate tracked_builds from the localization-admin.
    Indented to use with validate() as we assume that
    there is a dict with a tracked ota build in a format:
        {
          "version": "8.1",
          "branch": "dev"
        }
    """
    version = String(required=True)
    branch = String(required=True)


class OtaUpdateValidator(Schema):
    serial = String(required=False)
    device_id = String(required=False)


class OtaHeadersValidator(HeadersValidator):
    build_fingerprint = String(
        required=True,
        load_from=BUILD_FIELD,
        validate=Regexp(parsers.BUILD_FINGERPRINT_REGEXP)
    )


class FileValidator(Schema):
    filename = String(required=True)
    stream = Raw(required=True)

    @pre_load
    def load_file(self, data, **kwargs):
        if 'file' not in data:
            raise ValidationError('File not provided')
        data['filename'] = data['file'].filename
        data['stream'] = tempfile.SpooledTemporaryFile(max_size=500*1024)
        data['file'].save(data['stream'], buffer_size=16*1024*1024)
        data['stream'].seek(0)
        return data


class UploadValidator(FileValidator):
    branch = String(required=True)
    overwrite = Boolean(required=False, default=False)


class UpdateUploadValidator(UploadValidator):
    app_name = String(required=True)
    version_name = String(required=True)
    version_code = Integer(required=True, strict=True,
                           validate=Range(min=0, error='Version code must be non-negative integer'))
    package_name = String(required=True)

    @validates('filename')
    def validate_filename(self, value):
        if not value.lower().endswith('.apk'):
            raise ValidationError('File extension is not .apk')


class OtaUploadValidator(UploadValidator):
    os_version = String(required=True)
    build_version = String(required=True)
    manufacturer = String(required=True)
    product_name = String(required=True)
    model = String(required=True)
    note = String(required=False)


class ChangelogValidator(Schema):
    title = String(required=False, missing='')
    description = String(required=False)

    @validates_schema
    def validate_changelog(self, data):
        if data.get('title') and not data.get('description'):
            raise ValidationError('Description is not provided')
