import logging

from flask import request, Response
from flask.views import MethodView

from yaphone.utils.parsers import parsers

from yaphone.newpdater.src.common import localization, serialization, validation, views
from yaphone.newpdater.src.updates import firmware_service, serializers, types, updates_service, validators

logger = logging.getLogger(__name__)


class UpdatesView(MethodView):
    def get(self, **_):
        device_id, locale, update_types = validation.validate(validators.UpdateValidator, request.args, unpack=True)
        headers = validation.validate(validation.DeviceHeadersValidator, request.headers)

        device_type = parsers.UaParser.parse(
            headers['user_agent']
        ).get('device_type')

        language = validators.validate_language(request.accept_languages.best, locale)

        factory = localization.ConfigFactory(locale=language, **headers)
        response = []
        if types.UpdateType.GOOGLE_PLAY in update_types:
            market_updates = updates_service.get_market_updates(device_id, language)
            response += serialization.serialize_many(serializers.MarketUpdateSerializer, market_updates)

        if types.UpdateType.LOCAL in update_types:
            if device_type:
                local_updates = updates_service.get_local_updates_for_device(
                    device_type, factory.server_config, factory.translator
                )
            else:
                logger.info('Got a legacy client! device_id="%s"', device_id)
                local_updates = updates_service.get_local_updates_legacy(
                    factory.server_config, factory.translator
                )
            response += serialization.serialize_many(serializers.LocalUpdateSerializer, local_updates)

        return views.JsonResponse(response)


class UpdatesV3View(MethodView):
    def post(self, **_):
        params = request.get_json()
        headers = validation.validate(validation.DeviceHeadersValidator, request.headers)
        device_type = parsers.UaParser.parse(
            headers['user_agent']
        ).get('device_type')

        language = validators.validate_language(request.accept_languages.best, None)

        factory = localization.ConfigFactory(locale=language, **headers)

        local_updates = updates_service.get_local_updates_for_device(
            device_type, factory.server_config, factory.translator
        )

        installed_apps_versions = {app['package_name']: app['version_code'] for app in params['installed_apps']}

        filtered_updates = []
        for update in local_updates:
            if update.package_name in installed_apps_versions and \
                    update.version_code > installed_apps_versions[update.package_name]:
                filtered_updates.append(update)

        response = serialization.serialize(serializers.UpdatesV3Serializer, {
            'updates': filtered_updates,
        })

        return views.JsonResponse(response)


class OtaView(MethodView):
    response_serializer = serializers.OtaSerializer

    def get(self):
        serial, device_id = validation.validate(validators.OtaUpdateValidator, request.args, unpack=True)
        headers = validation.validate(validators.OtaHeadersValidator, request.headers)
        fingerprint = headers['build_fingerprint']
        device = parsers.FingerprintParser.parse(fingerprint)
        product = types.Product(**device)
        factory = localization.ConfigFactory(serial_number=serial, **headers)
        update = firmware_service.get_ota_update(factory.server_config, product)
        if update:
            response = serialization.serialize(self.response_serializer, update)
            logger.info('[OTA] response: %s', response)
            return views.JsonResponse(response)
        return views.JsonResponse({})


class OtaV3View(OtaView):
    response_serializer = serializers.OtaV3Serializer

    def get(self):
        serial, device_id = validation.validate(validators.OtaUpdateValidator, request.args, unpack=True)
        headers = validation.validate(validators.OtaHeadersValidator, request.headers)
        fingerprint = headers['build_fingerprint']
        device = parsers.FingerprintParser.parse(fingerprint)
        product = types.Product(**device)
        factory = localization.ConfigFactory(serial_number=serial, **headers)
        update = firmware_service.get_ota_update(factory.server_config, product)
        if update:
            force_update = factory.client_config.get_object('fota.force_update', False, log_missing=False)
            if force_update:
                update.priority = types.PriorityEnum.FORCED
            response = serialization.serialize(self.response_serializer, update)
            logger.info('[OTA] response: %s', response)
            return views.JsonResponse(response)
        return views.JsonResponse({})


class ListOtaView(MethodView):
    response_serializer = serializers.OtaSerializerNoChangelog

    def get(self):
        updates = []
        for update in firmware_service.get_ota_updates():
            update_data = serialization.serialize(self.response_serializer, update)
            updates.append(update_data)

        logger.debug('[OTA] response: %s', updates)

        return views.JsonResponse(updates)


class AppUploadView(MethodView):
    validator = validators.UpdateUploadValidator
    service = updates_service

    def post(self):
        data = validation.validate(self.validator, {**request.values, **request.files})
        self.service.save(data)
        return Response()


class OtaUploadView(MethodView):
    validator = validators.OtaUploadValidator
    service = firmware_service

    def post(self):
        data = validation.validate(self.validator, {**request.values, **request.files})
        self.service.save(data)
        return Response()
