#!/usr/bin/env python
# -*- coding: UTF-8 -*-
import os
import sys
sys.path.insert(1, os.path.join(sys.path[0], '../..'))

from nile.api.v1 import clusters, aggregators as na, extractors as ne, statface as ns
from nile.files import LocalFile
from qb2.api.v1 import filters as qf, extractors as qe
from common import (
    APPMETRIKA_LOG_PATH,
    DATES_LIST,
    JOB_ROOT,
    LAUNCHER_APIKEY,
    MAGIC_DATE_RANGE_STRING,
    USERNAME,
    get_clid1,
    validate_unicode,
    get_stat_client)

yt_token = os.environ['YT_TOKEN']
output_table = '$job_root/events'

common_path = os.path.join(sys.path[1], 'common.py')
cluster = clusters \
    .Hahn(pool='{}'.format(USERNAME), token=yt_token) \
    .env(
        templates=dict(job_root=JOB_ROOT, date=MAGIC_DATE_RANGE_STRING),
        packages=[LocalFile(common_path)],
        yt_spec_defaults=dict(
            pool_trees=["physical"],
            tentative_pool_trees=["cloud"]
        )
    )
job = cluster.job()
log = job.table(APPMETRIKA_LOG_PATH)


records = log.qb2(
    log='metrika-mobile-log',
    fields=['device_id', 'event_date', 'event_name',
        qe.log_field('Clids_Names', default="[]").rename('clids_names'),
        qe.log_field('Clids_Values', default="[]").rename('clids_values'),
        qe.log_field('Manufacturer', default='unknown'),
        qe.log_field('Model', default='unknown'),
    ],
    filters=[
        qf.equals('api_key_str', LAUNCHER_APIKEY),
        qf.equals('session_type', 'SESSION_FOREGROUND'),
        qf.defined('event_name'),
        qf.one_of('event_type', {'EVENT_CLIENT', 'EVENT_STATBOX'}),
        qf.one_of('event_date', DATES_LIST),
        qf.region_belongs(master_regions=[225], dictionary='Geobase', field='geo_id'),
    ]) \
    .project(
        'device_id',
        event='event_name',
        Manufacturer=ne.custom(validate_unicode, 'Manufacturer'),
        Model=ne.custom(validate_unicode, 'Model'),
        fielddate='event_date',
        clid=ne.custom(get_clid1, 'clids_names', 'clids_values')
    ) \
    .project(
        'device_id',
        'fielddate',
        qe.unfold_with_total('clid1', 'clid', total='_total_'),
        qe.unfold_with_total('event_name', 'event', total='_total_'),
        qe.unfold_with_total('manufacturer', 'Manufacturer', total='_total_'),
        qe.unfold_with_total('model', 'Model', total='_total_'),
    ) \
    .groupby('fielddate', 'clid1', 'event_name', 'manufacturer', 'model') \
    .aggregate(
        events=na.count(),
        devices=na.count_distinct('device_id', in_memory=False)
    ) \
    .sort('fielddate', 'devices') \
    .put(output_table)
job.run()


report = (
    ns.StatfaceReport()
    .path('Mobile_Soft_Launcher/appmetrika_events')
    .title('appmetrika_events')
    .scale('daily')
    .dimensions(
        ns.Date('fielddate').replaceable(),
        ns.StringSelector('clid1').replaceable(),
        ns.StringSelector('event_name').replaceable(),
        ns.StringSelector('manufacturer').replaceable(),
        ns.StringSelector('model').replaceable()
    )
    .measures(
        ns.Integer('devices').title('devices'),
        ns.Integer('events').title('events')
    )
)

client = get_stat_client()
(
    report.client(client)
    .data(records.read())
    .publish()
)
